/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.tools.util;

import org.apache.commons.io.IOUtils;
import org.junit.Test;
import org.mule.munit.tools.assertion.MunitToolsTestCase;
import org.mule.runtime.api.event.Event;
import org.mule.runtime.api.message.Message;
import org.mule.runtime.api.metadata.DataType;

import java.util.ArrayList;
import java.util.LinkedHashMap;

import static java.nio.charset.StandardCharsets.UTF_8;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.MatcherAssert.assertThat;

public class MuleComponentsTestCase extends MunitToolsTestCase {

  private static final String SIMPLE_MESSAGE_VALUE = "TEST";
  private static final String SIMPLE_MESSAGE_ATTRIBUTES_VALUE = "example.json";
  private static final String SIMPLE_MESSAGE_ATTRIBUTES_KEY = "fileName";
  private static final String LIST_MESSAGE_FIRST_VALUE = "ITEM1";
  private static final String LIST_MESSAGE_SECOND_VALUE = "ITEM2";

  @Override
  protected String getConfigFile() {
    return "util/get-mule-resource-functions.xml";
  }

  @Test
  public void createMessage() throws Exception {
    Event result = runFlow("createMessage");
    assertThat(result.getMessage().getPayload().getValue() instanceof Message, equalTo(true));
    Message message = (Message) result.getMessage().getPayload().getValue();
    assertThat(message.getPayload().getDataType().getMediaType(), equalTo(DataType.TEXT_STRING.getMediaType()));
    assertThat(message.getPayload().getValue(), equalTo(SIMPLE_MESSAGE_VALUE));
  }

  @Test
  public void createSimpleMessage() throws Exception {
    Event result = runFlow("createSimpleMessage");
    assertThat(result.getMessage().getPayload().getValue() instanceof Message, equalTo(true));
    Message message = (Message) result.getMessage().getPayload().getValue();
    assertThat(message.getPayload().getDataType().getMediaType(), equalTo(DataType.MULE_MESSAGE.getMediaType()));
    assertThat(message.getPayload().getValue(), equalTo(SIMPLE_MESSAGE_VALUE));
  }

  @Test
  public void createSimpleMessageWithAttributes() throws Exception {
    Event result = runFlow("createSimpleMessageWithAttributes");
    assertThat(result.getMessage().getPayload().getValue() instanceof Message, equalTo(true));
    Message message = (Message) result.getMessage().getPayload().getValue();
    assertThat(message.getPayload().getDataType().getMediaType(), equalTo(DataType.MULE_MESSAGE.getMediaType()));
    assertThat(message.getPayload().getValue(), equalTo(SIMPLE_MESSAGE_VALUE));
    assertThat(message.getAttributes().getValue() instanceof LinkedHashMap, equalTo(true));
    LinkedHashMap attributesValue = (LinkedHashMap) message.getAttributes().getValue();
    assertThat(attributesValue.get(SIMPLE_MESSAGE_ATTRIBUTES_KEY), equalTo(SIMPLE_MESSAGE_ATTRIBUTES_VALUE));
  }

  @Test
  public void createMessageWithAttributes() throws Exception {
    Event result = runFlow("createMessageWithAttributes");
    assertThat(result.getMessage().getPayload().getValue() instanceof Message, equalTo(true));
    Message message = (Message) result.getMessage().getPayload().getValue();
    assertThat(message.getPayload().getDataType().getMediaType(), equalTo(DataType.TEXT_STRING.getMediaType()));
    assertThat(message.getPayload().getValue(), equalTo(SIMPLE_MESSAGE_VALUE));
    assertThat(message.getAttributes().getValue() instanceof LinkedHashMap, equalTo(true));
    LinkedHashMap attributesValue = (LinkedHashMap) message.getAttributes().getValue();
    assertThat(attributesValue.get(SIMPLE_MESSAGE_ATTRIBUTES_KEY), equalTo(SIMPLE_MESSAGE_ATTRIBUTES_VALUE));
  }

  @Test
  public void createMessageWithMimeTypeAttributes() throws Exception {
    Event result = runFlow("createMessageWithMimeTypeAttributes");
    assertThat(result.getMessage().getPayload().getValue() instanceof Message, equalTo(true));
    Message message = (Message) result.getMessage().getPayload().getValue();
    assertThat(message.getPayload().getDataType().getMediaType(), equalTo(DataType.TEXT_STRING.getMediaType()));
    assertThat(message.getPayload().getValue(), equalTo(SIMPLE_MESSAGE_VALUE));
    LinkedHashMap attributesValue = (LinkedHashMap) message.getAttributes().getValue();
    assertThat(attributesValue.get(SIMPLE_MESSAGE_ATTRIBUTES_KEY), equalTo(SIMPLE_MESSAGE_ATTRIBUTES_VALUE));
    assertThat(message.getAttributes().getDataType().getMediaType(), equalTo(DataType.JSON_STRING.getMediaType()));
  }

  @Test
  public void createMessageFromFile() throws Exception {
    String exampleString = IOUtils.toString(getClass().getClassLoader().getResource("util/files/example.json"), UTF_8);
    Event result = runFlow("createMessageFromFile");
    assertThat(result.getMessage().getPayload().getValue() instanceof Message, equalTo(true));
    Message message = (Message) result.getMessage().getPayload().getValue();
    assertThat(message.getPayload().getDataType().getMediaType(), equalTo(DataType.JSON_STRING.getMediaType()));
    assertThat(message.getPayload().getValue(), equalTo(exampleString));
  }

  @Test
  public void createListMessages() throws Exception {
    Event result = runFlow("createListMessage");
    ArrayList payloadValue = (ArrayList) result.getMessage().getPayload().getValue();
    assertThat(payloadValue.size(), equalTo(2));

    assertThat(payloadValue.get(0) instanceof Message, equalTo(true));
    Message firstMessage = (Message) payloadValue.get(0);
    assertThat(firstMessage.getPayload().getDataType().getMediaType(), equalTo(DataType.TEXT_STRING.getMediaType()));
    assertThat(firstMessage.getPayload().getValue(), equalTo(LIST_MESSAGE_FIRST_VALUE));

    assertThat(payloadValue.get(1) instanceof Message, equalTo(true));
    Message secondMessage = (Message) payloadValue.get(1);
    assertThat(secondMessage.getPayload().getDataType().getMediaType(), equalTo(DataType.TEXT_STRING.getMediaType()));
    assertThat(secondMessage.getPayload().getValue(), equalTo(LIST_MESSAGE_SECOND_VALUE));
  }
}
