/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage;


import org.apache.commons.lang.StringUtils;
import org.mule.api.MuleContext;
import org.mule.api.context.notification.MessageProcessorNotificationListener;
import org.mule.api.processor.DefaultMessageProcessorPathElement;
import org.mule.api.processor.MessageProcessor;
import org.mule.config.spring.factories.SubflowMessageProcessorChainFactoryBean;
import org.mule.construct.AbstractPipeline;
import org.mule.context.notification.MessageProcessorNotification;
import org.mule.munit.assertion.processors.MunitFlow;
import org.mule.munit.plugins.coverage.listeners.Listeners;
import org.mule.processor.chain.InterceptingChainLifecycleWrapper;
import org.mule.processor.chain.SubflowInterceptingChainLifecycleWrapper;
import org.mule.routing.ChoiceRouter;
import org.mule.util.NotificationUtils;

import java.util.*;

//Mule ensure backwards until 4.x org.mule.construct.AbstractPipeline;
//Mule ensure backwards until 4.x org.mule.context.notification.MessageProcessorNotification;

// We just name it as long as they don't change package we should be good org.mule.processor.chain.InterceptingChainLifecycleWrapper
// We just name it as long as they don't change package we should be good org.mule.routing.ChoiceRouter

public class CoberturaModule {
    private static final String SUB_PROCESSORS_TOKEN = "subprocessors";

    private MuleContext context;
    private Reporter reporter = new Reporter();


    public CoberturaModule(MuleContext context) {
        this.context = context;
        this.reporter = new Reporter();
        this.initialise();
    }

    public Reporter getReporter() {
        return reporter;
    }

    private void initialise() {
        // Setup Context
        context.getNotificationManager().setNotificationDynamic(true);
        context.getNotificationManager().addInterfaceToType(MessageProcessorNotificationListener.class, MessageProcessorNotification.class);
        context.getNotificationManager().addListener(Listeners.getMessageProcessorListener(reporter));
        context.getNotificationManager().addListener(Listeners.getComponentFiringNotificationListener(reporter));

        // Set up reporter
        reporter.addFlowPaths(buildFlowPathsMap().values());
        reporter.addSubFlowPaths(buildSubFlowPathsMap().values());
    }

    private Map<MessageProcessor, String> buildFlowPathsMap() {
        Map<MessageProcessor, String> pathsMap = new HashMap<MessageProcessor, String>();

        Collection<AbstractPipeline> allFlows = context.getRegistry().lookupObjects(AbstractPipeline.class);
        for (AbstractPipeline flow : allFlows) {
            if (flow instanceof MunitFlow) {
                continue;
            }

            Map<MessageProcessor, String> paths = buildPathsMap(flow);
            reformatFlowRefToSubFlowPaths(paths);

            pathsMap.putAll(paths);
        }
        return pathsMap;
    }

    private Map<MessageProcessor, String> buildSubFlowPathsMap() {
        Map<MessageProcessor, String> pathsMap = new HashMap<MessageProcessor, String>();

        Collection<SubflowMessageProcessorChainFactoryBean> allSubFlows = context.getRegistry().lookupObjects(SubflowMessageProcessorChainFactoryBean.class);
        for (Object subFlow : allSubFlows) {
            SubflowInterceptingChainLifecycleWrapper sf = handleMunitSubFlowFromRegistry(subFlow);

            Map<MessageProcessor, String> paths = buildPathsMap(sf);

            reformatSubFlowsPaths(paths);

            pathsMap.putAll(filterOutFlowRefInSubFlow2SubFlow(paths));
        }

        return pathsMap;
    }

    private Map<MessageProcessor, String> buildPathsMap(Object flow) {
        DefaultMessageProcessorPathElement messageProcessorPathElement;
        if (AbstractPipeline.class.isAssignableFrom(flow.getClass())) {
            messageProcessorPathElement = buildMessageProcessorPathElement((AbstractPipeline) flow);
        } else {
            messageProcessorPathElement = buildMessageProcessorPathElement((SubflowInterceptingChainLifecycleWrapper) flow);
        }

        Map<MessageProcessor, String> filteredSubFlowPaths = filterOutInvalidChoiceMessageProcessors(NotificationUtils.buildPaths(messageProcessorPathElement));
        return filteredSubFlowPaths;
    }

    private DefaultMessageProcessorPathElement buildMessageProcessorPathElement(AbstractPipeline flow) {
        DefaultMessageProcessorPathElement pipeLinePathElement = new DefaultMessageProcessorPathElement(null, flow.getName());
        flow.addMessageProcessorPathElements(pipeLinePathElement);

        return pipeLinePathElement;
    }

    private DefaultMessageProcessorPathElement buildMessageProcessorPathElement(SubflowInterceptingChainLifecycleWrapper flow) {
        DefaultMessageProcessorPathElement pipeLinePathElement = new DefaultMessageProcessorPathElement(null, flow.getName());
        flow.addMessageProcessorPathElements(pipeLinePathElement);

        return pipeLinePathElement;
    }

    /**
     * This method purpose is to cast the subFlow to something that can be feed to buildPathsMap.
     * When asking the registry for subFlows, during a MUnit run  the returned objects are SubflowInterceptingChainLifecycleWrapper.
     * When asking the registry for subFlows, during a normal run the returned objects are SubflowMessageProcessorChainFactoryBean.
     * <p/>
     * In the second case we need to actually build the subFlow instance. In the first one not.
     *
     * @param subFlow
     * @return
     */
    private SubflowInterceptingChainLifecycleWrapper handleMunitSubFlowFromRegistry(Object subFlow) {
        // Handle sub-flows manage by munit
        SubflowInterceptingChainLifecycleWrapper sf;
        try {
            if (SubflowInterceptingChainLifecycleWrapper.class.isAssignableFrom(subFlow.getClass())) {
                sf = (SubflowInterceptingChainLifecycleWrapper) subFlow;
            } else {
                sf = (SubflowInterceptingChainLifecycleWrapper) ((SubflowMessageProcessorChainFactoryBean) subFlow).getObject();
            }
            return sf;
        } catch (Exception e) {
            throw new RuntimeException("MUnit Cobertura Plugin, there was an error trying to account for sub-flows.", e);
        }
    }

    /**
     * Remove the first 2 tokens from the path as it add no value for registering purposes.
     *
     * @param filteredSubFlowPaths
     */
    private void reformatSubFlowsPaths(Map<MessageProcessor, String> filteredSubFlowPaths) {
        for (Map.Entry<MessageProcessor, String> entry : filteredSubFlowPaths.entrySet()) {
            String[] tokens = entry.getValue().split("/");

            List segments = new ArrayList(Arrays.asList(tokens));
            segments.remove(0);
            segments.remove(0);

            String realPath = "/" + StringUtils.join(segments, "/");

            entry.setValue(realPath);
        }
    }


    /**
     * We need to reformat the flow-ref paths to subFlows.
     * The path will expand and show each MP in the subFlow.
     * Instead of that we just register the unique part of the the flowRef to a subFlow.
     * <p/>
     * If we get this:
     * /flowName/processors/index/subFlowName/subprocessors/index
     * <p/>
     * We only register this:
     * /flowName/processors/index/subFlowName
     *
     * @param paths
     */

    private void reformatFlowRefToSubFlowPaths(Map<MessageProcessor, String> paths) {
        for (Map.Entry<MessageProcessor, String> entry : paths.entrySet()) {
            String path = entry.getValue();
            List segments = new ArrayList(Arrays.asList(path.split("/")));

            if (segments.size() >= 6 && segments.get(5).equals(SUB_PROCESSORS_TOKEN)) {
                String realPath = StringUtils.join(segments.subList(0, 4), "/");
                entry.setValue(realPath);
            }
        }
    }

    /**
     * Take out flow-ref to subFlow paths.
     * <p/>
     * If there is more than one subprocessors in the path is a flow-ref to a subflow.
     *
     * @param subFlowPaths
     * @return
     */
    private Map<MessageProcessor, String> filterOutFlowRefInSubFlow2SubFlow(Map<MessageProcessor, String> subFlowPaths) {
        Map<MessageProcessor, String> filteredPaths = new HashMap();

        for (MessageProcessor k : subFlowPaths.keySet()) {
            String path = subFlowPaths.get(k);
            if (!(StringUtils.countMatches(path, SUB_PROCESSORS_TOKEN) > 1)) {
                filteredPaths.put(k, path);
            }
        }

        return filteredPaths;
    }

    /**
     * The two MP classes below are part of the choice router chain.
     * We filter out the ChoiceRouter because if present it will always be called. But it's considered as one more MP
     * thus it reflects wrong in the coverage %.
     * <p/>
     * We filter out InterceptingChainLifecycleWrapper because although is present, between creation of the flow and <p/>
     * actually run it the flow the instance of that class in the chain is recreated. That is, the instance change but <p/>
     * the content doesn't. Because of that, the notification never gets triggered and is as the MP never gets called <p/>
     * although it is.
     *
     * @param messageProcessors
     * @return a list of the mp without the specific ones for choice
     */
    private Map<MessageProcessor, String> filterOutInvalidChoiceMessageProcessors(Map<MessageProcessor, String> messageProcessors) {
        Map<MessageProcessor, String> mpMap = new HashMap<MessageProcessor, String>();

        for (Map.Entry<MessageProcessor, String> mp : messageProcessors.entrySet()) {
            if (!(InterceptingChainLifecycleWrapper.class.isAssignableFrom(mp.getKey().getClass())) && !(ChoiceRouter.class.isAssignableFrom(mp.getKey().getClass()))) {
                mpMap.put(mp.getKey(), mp.getValue());
            }
        }
        return mpMap;
    }
}
