/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage;

import java.util.*;

public class CoverageCalculator {

    private Set<String> coveredPaths;
    private Set<String> flows;
    private Set<String> subFlows;
    private Set<String> batches;

    private static class Count {
        int totalMp;
        int coveredMp;
    }

    public static class CoverageResult {
        private final Double coverage;
        private final Map<String, Double> containersCoverage;

        public CoverageResult(Double coverage, Map<String, Double> containersCoverage) {
            this.coverage = coverage;
            this.containersCoverage = containersCoverage;
        }

        public Double getCoverage() {
            return coverage;
        }

        public Map<String, Double> getContainersCoverage() {
            return containersCoverage;
        }
    }

    public CoverageCalculator(Set<String> coveredPaths, Set<String> flows, Set<String> subFlows, Set<String> batches) {
        this.coveredPaths = coveredPaths;

        this.flows = flows;
        this.subFlows = subFlows;
        this.batches = batches;
    }

    public CoverageResult calculate() {
        Map<String, Count> pathPerFlow = new HashMap<String, Count>();

        countFlowPaths(pathPerFlow);
        countSubFlowPaths(pathPerFlow);
        countBatchPaths(pathPerFlow);

        Double coverage = percentage(coveredPaths.size(), flows.size() + subFlows.size() + batches.size());

        Map<String, Double> containersCoverage = new HashMap<String, Double>();
        for (Map.Entry<String, Count> count : pathPerFlow.entrySet()) {
            containersCoverage.put(count.getKey(), percentage(count.getValue().coveredMp, count.getValue().totalMp));
        }

        return new CoverageResult(coverage, containersCoverage);
    }

    private void countFlowPaths(Map<String, Count> pathPerFlow) {
        List<String> tokens = new ArrayList<String>();
        tokens.add("/" + PathBuilder.PROCESSORS_TOKEN);
        for (String path : flows) {
            countPath(path, pathPerFlow, tokens);
        }
    }

    private void countSubFlowPaths(Map<String, Count> pathPerFlow) {
        List<String> tokens = new ArrayList<String>();
        tokens.add("/" + PathBuilder.SUB_PROCESSORS_TOKEN);
        for (String path : subFlows) {
            countPath(path, pathPerFlow, tokens);
        }
    }

    private void countBatchPaths(Map<String, Count> pathPerFlow) {
        List<String> tokens = new ArrayList<String>();
        tokens.add("/" + PathBuilder.BATCH_INPUT_TOKEN);
        tokens.add("/" + PathBuilder.BATCH_PROCESS_RECORD_TOKEN);
        tokens.add("/" + PathBuilder.BATCH_ON_COMPLETE_TOKEN);
        for (String path : batches) {
            countPath(path, pathPerFlow, tokens);
        }
    }

    private void countPath(String path, Map<String, Count> pathPerFlow, List<String> tokenFlags) {
        String containerName = buildContainerName(path, tokenFlags);
        if (!pathPerFlow.containsKey(containerName)) {
            pathPerFlow.put(containerName, new Count());
        }
        Count count = pathPerFlow.get(containerName);
        count.totalMp++;
        if (coveredPaths.contains(path)) {
            count.coveredMp++;
        }
    }

    private String buildContainerName(String path, List<String> tokenFlags) {
        String containerName = null;
        for (String tokenFlag : tokenFlags) {
            try {
                containerName = path.substring(0, path.indexOf(tokenFlag));
                break;
            } catch (IndexOutOfBoundsException e) {
                continue;
            }
        }
        return containerName;
    }

    private double percentage(int size, int size1) {
        if (size1 != 0) {
            return ((double) size / size1) * 100;
        }
        return 0;
    }


}
