/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage;

import org.apache.commons.lang.StringUtils;
import org.mule.munit.plugins.coverage.report.SingleRunReport;

import java.util.*;

public class Reporter {

    private Set<String> subFlowPaths = new HashSet();
    private Set<String> flowPaths = new HashSet<String>();
    private Set<String> batchPaths = new HashSet<String>();

    private Set<String> coveredPaths = new HashSet<String>();

    public void addCoveredPath(String path) {
        // Normal Path of any flow
        if (flowPaths.contains(path)) {
            coveredPaths.add(path);
            return;
        }

        if (batchPaths.contains(path)) {
            coveredPaths.add(path);
            return;
        }

        markPathsInsideGlobalCatchsAsCovered(path);

        markFlowRefToSubFlowsAsCovered(path);

        markSubFlowPathsAsCovered(path);
    }

    /**
     * The paths of a global catch exception strategy needs pre-processing
     * As they come appended with the name of the flow from where they are being executed.
     *
     * @param path
     */
    private void markPathsInsideGlobalCatchsAsCovered(String path) {
        List<String> segments = Arrays.asList(path.split("/"));

        if (segments.size() >= 3 && "es".equals(segments.get(3))) {
            String reformattedPath = "/" + StringUtils.join(segments.subList(2, segments.size()), "/");
            for (String p : flowPaths) {
                if (reformattedPath.equals(p)) {
                    coveredPaths.add(p);
                    break;
                }
            }
        }
    }

    /**
     * We save this path only as it represents the flow ref
     * The path will represent a sub-flow MP itself
     *
     * @param path
     */
    private void markFlowRefToSubFlowsAsCovered(String path) {
        // Flow ref to sub-flows in a flow
        for (String p : flowPaths) {
            if (path.startsWith(p)) {
                coveredPaths.add(p);
                //  we do not return the method
                // for this flags the flow ref as coveredPaths
                // but we still need to process the sub flow path
            }
        }

        // Flow ref to sub-flows in a sub-flow
        for (String p : subFlowPaths) {
            if (path.contains(p) && !path.endsWith(p)) {
                coveredPaths.add(p);
                //  we do not return the method
                // for this flags the flow ref as coveredPaths
                // but we still need to process the sub flow path
            }
        }
    }

    /**
     * We save p for this represents the unique sub-flow mp
     * independently from the flow/sub-flow who referenced it
     *
     * @param path
     */
    private void markSubFlowPathsAsCovered(String path) {
        for (String p : subFlowPaths) {
            if (path.endsWith(p)) {
                coveredPaths.add(p);
                break;
            }
        }
    }

    public void addFlowPaths(Collection<String> mps) {
        flowPaths.addAll(mps);
    }

    public void addSubFlowPaths(Collection<String> mps) {
        subFlowPaths.addAll(mps);
    }

    public void addBatchPaths(Collection<String> mps) {
        batchPaths.addAll(mps);
    }

    public SingleRunReport getReport() {
        CoverageCalculator calculator = new CoverageCalculator(coveredPaths, flowPaths, subFlowPaths, batchPaths);
        CoverageCalculator.CoverageResult result = calculator.calculate();

        Set<String> allPaths = new HashSet<String>();
        allPaths.addAll(flowPaths);
        allPaths.addAll(subFlowPaths);
        allPaths.addAll(batchPaths);

        return new SingleRunReport(result.getCoverage(), result.getContainersCoverage(), coveredPaths, allPaths);
    }
}
