/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage.report;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.munit.plugins.coverage.CoverageCalculator;
import org.mule.munit.plugins.coverage.PathBuilder;
import org.mule.munit.plugins.coverage.report.model.MuleFlow;
import org.mule.munit.plugins.coverage.report.model.MuleResource;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URL;
import java.util.*;

public class CoverageReportBuilder {
    private transient Log log = LogFactory.getLog(this.getClass());

    private Set<String> flowsToIgnore = new HashSet<String>();
    private String applicationResources;

    public CoverageReportBuilder(String applicationResources) {
        this.applicationResources = applicationResources;
    }

    public void setFlowsToIgnore(Set<String> flowsToIgnore) {
        this.flowsToIgnore = flowsToIgnore;
    }

    public ApplicationCoverageReport buildReport(Set<String> coveredPaths, Set<String> appFlowPaths, Set<String> appSubFlowPaths, Set<String> appBatchPaths) {

        coveredPaths = filterIgnoredFlows(coveredPaths);
        appFlowPaths = filterIgnoredFlows(appFlowPaths);
        appSubFlowPaths = filterIgnoredFlows(appSubFlowPaths);
        appBatchPaths = filterIgnoredFlows(appBatchPaths);


        Map<String, List<String>> filesMap = buildFilesFlowMap(applicationResources, appFlowPaths);

        removeGlobalCatchFromFilesFlowMapIfThereAreNoReferences(filesMap, appFlowPaths, appSubFlowPaths, appBatchPaths);

        Map<String, List<String>> flowsMap = buildApplicationFlowPathMap(appFlowPaths, appSubFlowPaths, appBatchPaths);

        CoverageCalculator calculator = new CoverageCalculator(coveredPaths, appFlowPaths, appSubFlowPaths, appBatchPaths);
        CoverageCalculator.CoverageResult coverageResult = calculator.calculate();

        Map<String, List<String>> flowsCoveredPaths = PathBuilder.buildFlowPathsMap(coveredPaths);

        int totalMpCount = 0;
        int coveredMpCount = 0;
        ApplicationCoverageReport report = new ApplicationCoverageReport();

        for (String fileName : filesMap.keySet()) {
            MuleResource muleResource = new MuleResource(fileName);

            List<String> flowsFromFile = filesMap.get(fileName);
            for (String flowName : flowsFromFile) {
                // Only Report flows that hasn't been ignored
                if (!flowsToIgnore.contains(flowName)) {
                    MuleFlow muleFlow = new MuleFlow(flowName);

                    if (flowsMap.containsKey(flowName)) {
                        muleFlow.getPaths().addAll(flowsMap.get(flowName));
                        totalMpCount += flowsMap.get(flowName).size();
                    }

                    if (flowsCoveredPaths.containsKey(flowName)) {
                        muleFlow.getCoveredPaths().addAll(flowsCoveredPaths.get(flowName));
                        coveredMpCount += flowsCoveredPaths.get(flowName).size();
                    }

                    muleResource.getFlows().add(muleFlow);
                }
            }
            report.getResources().add(muleResource);
        }
        report.setCoverage(((double)coveredMpCount*100)/totalMpCount);
        report.setResourcesWeight();

        return report;
    }

    private Set<String> filterIgnoredFlows(Set<String> paths) {
        if (null != flowsToIgnore) {
            return PathBuilder.filterPaths(paths, flowsToIgnore);
        }
        return paths;
    }

    private Map<String, List<String>> buildFilesFlowMap(String resources, Set<String> appFlowPaths) {
        Map<String, List<String>> filesMap = new HashMap<String, List<String>>();
        log.debug("Building Files flow map...");

        if (resources != null && !resources.equals("")) {
            for (String resource : resources.split(",")) {
                URL fileUrl = this.getClass().getClassLoader().getResource(resource);

                filesMap.put(resource, getFlowsFromFile(fileUrl));
            }
        }

        log.debug("Files flow map building done...");
        return filesMap;
    }

    /**
     * Remove those Flows name from each file list as long as they are not reported as path of the application.
     * Should only apply(filter) to global catch that are not referenced by any flow.
     *
     * @param filesMap
     * @param appFlowPaths
     */
    private void removeGlobalCatchFromFilesFlowMapIfThereAreNoReferences(Map<String, List<String>> filesMap, Set<String> appFlowPaths, Set<String> appSubFlowPaths, Set<String> appBatchPaths) {
        Set<String> flowNamesInPaths = new HashSet<String>();
        for (String p : appFlowPaths) {
            flowNamesInPaths.add(getFlowNameFromPath(p));
        }

        for (String p : appSubFlowPaths) {
            flowNamesInPaths.add(p.split("/")[1]);
        }

        for (String p : appBatchPaths) {
            flowNamesInPaths.add(p.split("/")[1]);
        }

        for (Map.Entry<String, List<String>> e : filesMap.entrySet()) {
            List<String> flows = new ArrayList<String>();
            for (String flow : e.getValue()) {
                if (flowNamesInPaths.contains(flow)) {
                    flows.add(flow);
                }
            }
            e.setValue(flows);
        }
    }

    private String getFlowNameFromPath(String flowPath) {
        String flowName;

        //Deal with flow names from apikit which contains :\\/
        if (flowPath.contains(":\\/")) {
            flowName = StringUtils.strip(StringUtils.join(flowPath.split("/"), "/", 0, 3), "/");
            flowName = flowName.replace(":\\/", ":/");
        } else {
            flowName = flowPath.split("/")[1];
        }

        return flowName;
    }

    private List<String> getFlowsFromFile(URL fileUrl) {
        log.debug("Getting flows from file [" + fileUrl + "]");

        List<String> flowNames = new ArrayList<String>();

        String xpathQuery = "//*[local-name()='flow' or local-name()='sub-flow' or local-name()='job' or local-name()='catch-exception-strategy' or local-name()='mapping-exception-strategy']";
        try {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            factory.setNamespaceAware(true);
            DocumentBuilder builder = factory.newDocumentBuilder();
            Document doc = builder.parse(fileUrl.openStream());
            XPath xpath = XPathFactory.newInstance().newXPath();

            NodeList nl = (NodeList) xpath.compile(xpathQuery).evaluate(doc, XPathConstants.NODESET);
            for (int i = 0; i < nl.getLength(); i++) {
                Node nameAttribute = nl.item(i).getAttributes().getNamedItem("name");
                if (null != nameAttribute) {
                    flowNames.add(nameAttribute.getNodeValue());
                }
            }

        } catch (ParserConfigurationException e) {
            log.debug("Error parsing file", e);
        } catch (SAXException e) {
            log.debug("Error parsing file", e);
        } catch (FileNotFoundException e) {
            log.debug("File not found", e);
        } catch (XPathExpressionException e) {
            log.debug("Error parsing file", e);
        } catch (IOException e) {
            log.debug("Error reading file", e);
        } finally {
            log.debug("Flow names loaded: " + flowNames);
            return flowNames;
        }
    }

    private Map<String, List<String>> buildApplicationFlowPathMap(Set<String> flows, Set<String> subFlows, Set<String> batches) {
        Map<String, List<String>> flowsMap = new HashMap<String, List<String>>();

        flowsMap.putAll(PathBuilder.buildFlowPathsMap(flows));
        flowsMap.putAll(PathBuilder.buildFlowPathsMap(subFlows));
        flowsMap.putAll(PathBuilder.buildFlowPathsMap(batches));

        return flowsMap;
    }
}
