/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.*;

public class CoverageCalculator {
    private transient Log log = LogFactory.getLog(this.getClass());


    private Set<String> coveredPaths;
    private Set<String> flows;
    private Set<String> subFlows;
    private Set<String> batches;

    private static class Count {
        int totalMp;
        int coveredMp;
    }

    public static class CoverageResult {
        private final Double coverage;
        private final Map<String, Double> containersCoverage;

        public CoverageResult(Double coverage, Map<String, Double> containersCoverage) {
            this.coverage = coverage;
            this.containersCoverage = containersCoverage;
        }

        public Double getCoverage() {
            return coverage;
        }

        public Map<String, Double> getContainersCoverage() {
            return containersCoverage;
        }
    }

    public CoverageCalculator(Set<String> coveredPaths, Set<String> flows, Set<String> subFlows, Set<String> batches) {
        this.coveredPaths = coveredPaths;

        this.flows = flows;
        this.subFlows = subFlows;
        this.batches = batches;

        //TODO: remove this after MULE-9996 gets fixed
        countParentAsCovered();
    }

    public CoverageResult calculate() {
        Map<String, Count> pathPerFlow = new HashMap<String, Count>();

        countFlowPaths(pathPerFlow);
        countSubFlowPaths(pathPerFlow);
        countBatchPaths(pathPerFlow);

        int totalPathCount = flows.size() + subFlows.size() + batches.size();
        Double contextCoverage = percentage(coveredPaths.size(), totalPathCount);

        Map<String, Double> containersCoverage = new HashMap<String, Double>();
        for (Map.Entry<String, Count> count : pathPerFlow.entrySet()) {
            containersCoverage.put(count.getKey(), percentage(count.getValue().coveredMp, count.getValue().totalMp));
        }

        log.debug("Coverage Calculations --> Covered Paths: " + coveredPaths.size() + " - " + "Total Paths: " + totalPathCount + " - Context Coverage: " + contextCoverage);
        return new CoverageResult(contextCoverage, containersCoverage);
    }

    private void countFlowPaths(Map<String, Count> pathPerFlow) {
        List<String> tokens = new ArrayList<String>();
        tokens.add("/" + PathBuilder.PROCESSORS_TOKEN);
        for (String path : flows) {
            countPath(path, pathPerFlow, tokens);
        }
    }

    private void countSubFlowPaths(Map<String, Count> pathPerFlow) {
        List<String> tokens = new ArrayList<String>();
        tokens.add("/" + PathBuilder.SUB_PROCESSORS_TOKEN);
        for (String path : subFlows) {
            countPath(path, pathPerFlow, tokens);
        }
    }

    private void countBatchPaths(Map<String, Count> pathPerFlow) {
        List<String> tokens = new ArrayList<String>();
        tokens.add("/" + PathBuilder.BATCH_INPUT_TOKEN);
        tokens.add("/" + PathBuilder.BATCH_PROCESS_RECORD_TOKEN);
        tokens.add("/" + PathBuilder.BATCH_ON_COMPLETE_TOKEN);
        for (String path : batches) {
            countPath(path, pathPerFlow, tokens);
        }
    }

    private void countPath(String path, Map<String, Count> pathPerFlow, List<String> tokenFlags) {
        String containerName = buildContainerName(path, tokenFlags);
        if (!pathPerFlow.containsKey(containerName)) {
            pathPerFlow.put(containerName, new Count());
        }
        Count count = pathPerFlow.get(containerName);
        count.totalMp++;
        if (coveredPaths.contains(path)) {
            count.coveredMp++;
        }
    }

    private String buildContainerName(String path, List<String> tokenFlags) {
        String containerName = null;
        for (String tokenFlag : tokenFlags) {
            try {
                containerName = path.substring(0, path.indexOf(tokenFlag));
                break;
            } catch (IndexOutOfBoundsException e) {
                continue;
            }
        }
        return containerName;
    }

    private double percentage(int part, int total) {
        if (total != 0) {
//            return ((double) size / size1) * 100;
            return ((double) part * 100) / total;
        }
        return 0;
    }


    private void countParentAsCovered() {
        Set<String> coveredParentPaths = new HashSet<String>();
        for (String cp : coveredPaths) {
            String parentPath = getParentPath(cp);
            if (StringUtils.isNotBlank(parentPath)) {
                if (flows.contains(parentPath) || subFlows.contains(parentPath) || batches.contains(parentPath)) {
                    coveredParentPaths.add(parentPath);
                }
            }
        }
        coveredPaths.addAll(coveredParentPaths);
    }

    private String getParentPath(String path) {
        String parentPath = "";
        if (path.contains(PathBuilder.PROCESSORS_TOKEN) || path.contains(PathBuilder.SUB_PROCESSORS_TOKEN) || path.contains(PathBuilder.EXCEPTION_TOKEN)) {
            String[] tokens = path.split("/");
            //Deal with flow names from apikit which contains :\\/
            if (path.contains(":\\/")) {
                if (tokens.length > 5) {
                    parentPath = StringUtils.strip(StringUtils.join(tokens, "/", 0, tokens.length - 1), "/");
                }
            } else {
                if (tokens.length > 4) {
                    parentPath = StringUtils.join(tokens, "/", 0, tokens.length - 1);
                }
            }
        }
        return parentPath;
    }

}
