/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */

package org.mule.munit.plugins.coverage.report;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.munit.plugins.coverage.path.PathFormatter;
import org.mule.munit.plugins.coverage.path.PathParser;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.URL;
import java.util.*;

public class FileFlowMap {
    private transient Log log = LogFactory.getLog(this.getClass());

    private Map<String, List<String>> filesMap;

    public Set<String> getFiles() {
        return filesMap.keySet();
    }

    public void build(String resources, Set<String> appFlowPaths, Set<String> appSubFlowPaths, Set<String> appBatchPaths) {
        this.filesMap = new HashMap<String, List<String>>();
        log.debug("Building Files flow map...");

        if (resources != null && !resources.equals("")) {
            for (String resource : resources.split(",")) {
                URL fileUrl = this.getClass().getClassLoader().getResource(resource);
                filesMap.put(resource, getFlowsFromFile(fileUrl));
            }
        }
        removeGlobalCatchIfThereAreNoReferences(appFlowPaths, appSubFlowPaths, appBatchPaths);

        log.debug("Files flow map building done...");
    }

    private List<String> getFlowsFromFile(URL fileUrl) {
        log.debug("Getting flows from file [" + fileUrl + "]");

        List<String> flowNames = new ArrayList<String>();
        String xpathQuery = "//*[local-name()='flow' or local-name()='sub-flow' or local-name()='job' or local-name()='catch-exception-strategy' or local-name()='mapping-exception-strategy']";
        try {
            NodeList nodeList = buildNodeList(fileUrl, xpathQuery);
            addFlowFromNodeList(nodeList, flowNames);
        } catch (ParserConfigurationException e) {
            log.debug("Error parsing file", e);
        } catch (SAXException e) {
            log.debug("Error parsing file", e);
        } catch (FileNotFoundException e) {
            log.debug("File not found", e);
        } catch (XPathExpressionException e) {
            log.debug("Error parsing file", e);
        } catch (IOException e) {
            log.debug("Error reading file", e);
        } finally {
            log.debug("Flow names loaded: " + flowNames);
            return flowNames;
        }
    }

    private NodeList buildNodeList(URL fileUrl, String xpathQuery) throws ParserConfigurationException, IOException, SAXException, XPathExpressionException {
        DocumentBuilder builder = createBuilder();
        Document doc = builder.parse(fileUrl.openStream());
        XPath xpath = XPathFactory.newInstance().newXPath();
        return (NodeList) xpath.compile(xpathQuery).evaluate(doc, XPathConstants.NODESET);
    }

    private DocumentBuilder createBuilder() throws ParserConfigurationException {
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        factory.setNamespaceAware(true);
        return factory.newDocumentBuilder();
    }

    private void addFlowFromNodeList(NodeList nodeList, List<String> flowNames) {
        for (int i = 0; i < nodeList.getLength(); i++) {
            Node nameAttribute = nodeList.item(i).getAttributes().getNamedItem("name");
            if (null != nameAttribute) {
                flowNames.add(PathFormatter.getFormattedFlowName(nameAttribute.getNodeValue()));
            }
        }
    }

    public List<String> get(String fileName) {
        return this.filesMap.get(fileName);
    }

    /**
     * Remove those Flows name from each file list as long as they are not reported as path of the application.
     * Should only apply(filter) to global catch that are not referenced by any flow.
     *
     * @param appFlowPaths
     */
    private void removeGlobalCatchIfThereAreNoReferences(Set<String> appFlowPaths, Set<String> appSubFlowPaths, Set<String> appBatchPaths) {
        Set<String> flowNamesInPaths = new HashSet<String>();

        addFlowNames(flowNamesInPaths, appFlowPaths);
        addFlowNames(flowNamesInPaths, appSubFlowPaths);
        addFlowNames(flowNamesInPaths, appBatchPaths);

        for (Map.Entry<String, List<String>> e : filesMap.entrySet()) {
            List<String> flows = new ArrayList<String>();
            for (String flow : e.getValue()) {
                if (flowNamesInPaths.contains(flow)) {
                    flows.add(flow);
                }
            }
            e.setValue(flows);
        }
    }

    private void addFlowNames(Set<String> flowNamesInPaths, Set<String> paths) {
        for (String p : paths) {
            flowNamesInPaths.add(PathParser.getFlowName(p));
        }
    }
}
