/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage.path;

import com.google.common.base.Function;
import org.mule.api.MuleContext;
import org.mule.api.processor.DefaultMessageProcessorPathElement;
import org.mule.api.processor.MessageProcessor;
import org.mule.api.processor.MessageProcessorContainer;
import org.mule.config.spring.factories.SubflowMessageProcessorChainFactoryBean;
import org.mule.construct.AbstractFlowConstruct;
import org.mule.construct.AbstractPipeline;
import org.mule.munit.assertion.processors.MunitFlow;
import org.mule.munit.common.util.ProxyExtractor;
import org.mule.processor.chain.InterceptingChainLifecycleWrapper;
import org.mule.processor.chain.SubflowInterceptingChainLifecycleWrapper;
import org.mule.util.NotificationUtils;

import java.util.*;

public class PathBuilder {

    private MuleContext context;

    public PathBuilder(MuleContext muleContext) {
        this.context = muleContext;
    }

    public <T> Map buildGenericPathsMap(Class<T> clazzType, Function<Object,Boolean> shouldAdd, Function<Object, Map> reformatPaths) {
        Map pathsMap = new HashMap<MessageProcessor, String>();
        Collection<T> muleRegistryObjects = context.getRegistry().lookupObjects(clazzType);
        for (Object muleRegistryObject : muleRegistryObjects) {
            Object flowConstruct = ProxyExtractor.extractIfProxy(muleRegistryObject);
            if (shouldAdd.apply(flowConstruct)) {
                pathsMap.putAll(reformatPaths.apply(flowConstruct));
            }
        }
        return pathsMap;
    }

    public Map<MessageProcessor, String> buildFlowPathsMap() {
        return buildGenericPathsMap(AbstractPipeline.class, new Function<Object, Boolean>() {
            public Boolean apply(Object flowConstruct) {
                return !(flowConstruct instanceof MunitFlow);
            }
        }, new Function<Object, Map>() {
            public Map apply(Object flowConstruct) {
                Map paths = buildPathsMap(flowConstruct);
                paths = PathFormatter.reformatFlowPaths(paths);
                return paths;
            }
        });
    }

    public Map<MessageProcessor, String> buildSubFlowPathsMap() {
        return buildGenericPathsMap(SubflowMessageProcessorChainFactoryBean.class, new Function<Object, Boolean>() {
            public Boolean apply(Object flowConstruct) {
                return true;
            }
        }, new Function<Object, Map>() {
            public Map apply(Object flowConstruct) {
                SubflowInterceptingChainLifecycleWrapper sf = handleMunitSubFlowFromRegistry(flowConstruct);
                Map paths = buildPathsMap(sf);
                paths = PathFormatter.reformatSubFlowPaths(paths);
                return filterOutFlowRefInSubFlow2SubFlow(paths);
            }
        });
    }

    public Map<MessageProcessor, String> buildBatchPathsMap() {
        return buildGenericPathsMap(AbstractFlowConstruct.class, new Function<Object, Boolean>() {
            public Boolean apply(Object flowConstruct) {
                return "Batch".equals(((AbstractFlowConstruct)flowConstruct).getConstructType());
            }
        }, new Function<Object, Map>() {
            public Map apply(Object flowConstruct) {
                Map paths = buildPathsMap(flowConstruct);
                paths = PathFormatter.reformatBatchPaths(paths);
                return paths;
            }
        });
    }

    private Map<MessageProcessor, String> buildPathsMap(Object flow) {
        String flowName;
        if (flow instanceof AbstractFlowConstruct) {
            flowName = ((AbstractFlowConstruct) flow).getName();
        } else {
            flowName = ((SubflowInterceptingChainLifecycleWrapper) flow).getName();
        }
        DefaultMessageProcessorPathElement messageProcessorPathElement = buildMessageProcessorPathElement((MessageProcessorContainer) flow, flowName);
        return filterOutInvalidChoiceMessageProcessors(NotificationUtils.buildPaths(messageProcessorPathElement));
    }

    private DefaultMessageProcessorPathElement buildMessageProcessorPathElement(MessageProcessorContainer flow, String flowName) {
        DefaultMessageProcessorPathElement pipeLinePathElement = new DefaultMessageProcessorPathElement(null, flowName);
        flow.addMessageProcessorPathElements(pipeLinePathElement);

        return pipeLinePathElement;
    }

    /**
     * This method purpose is to cast the subFlow to something that can be feed to buildPathsMap.
     * When asking the registry for subFlows, during a MUnit run  the returned objects are SubflowInterceptingChainLifecycleWrapper.
     * When asking the registry for subFlows, during a normal run the returned objects are SubflowMessageProcessorChainFactoryBean.
     * <p/>
     * In the second case we need to actually build the subFlow instance. In the first one not.
     *
     * @param subFlow
     * @return
     */
    private SubflowInterceptingChainLifecycleWrapper handleMunitSubFlowFromRegistry(Object subFlow) {
        // Handle sub-flows manage by munit
        SubflowInterceptingChainLifecycleWrapper sf;
        try {
            if (subFlow instanceof SubflowInterceptingChainLifecycleWrapper) {
                sf = (SubflowInterceptingChainLifecycleWrapper) subFlow;
            } else {
                sf = (SubflowInterceptingChainLifecycleWrapper) ((SubflowMessageProcessorChainFactoryBean) subFlow).getObject();
            }
            return sf;
        } catch (Exception e) {
            throw new RuntimeException("MUnit Cobertura Plugin, there was an error trying to account for sub-flows.", e);
        }
    }

    /**
     * Take out flow-ref to subFlow paths.
     * <p/>
     * If there is more than one subprocessors in the path is a flow-ref to a subflow.
     *
     * @param subFlowPaths
     * @return
     */
    private Map<MessageProcessor, String> filterOutFlowRefInSubFlow2SubFlow(Map<MessageProcessor, String> subFlowPaths) {
        Map<MessageProcessor, String> filteredPaths = new HashMap();

        for (MessageProcessor k : subFlowPaths.keySet()) {
            String path = subFlowPaths.get(k);
            if (PathParser.doesNotHaveFlowRefToSubFlow(path)) {
                filteredPaths.put(k, path);
            }
        }
        return filteredPaths;
    }

    /**
     * The two MP classes below are part of the choice router chain.
     * We filter out the ChoiceRouter because if present it will always be called. But it's considered as one more MP
     * thus it reflects wrong in the coverage %.
     * <p/>
     * We filter out InterceptingChainLifecycleWrapper because although is present, between creation of the flow and <p/>
     * actually run it the flow the instance of that class in the chain is recreated. That is, the instance change but <p/>
     * the content doesn't. Because of that, the notification never gets triggered and is as the MP never gets called <p/>
     * although it is.
     *
     * @param messageProcessors
     * @return a list of the mp without the specific ones for choice
     */
    private Map<MessageProcessor, String> filterOutInvalidChoiceMessageProcessors(Map<MessageProcessor, String> messageProcessors) {
        Map<MessageProcessor, String> mpMap = new HashMap<MessageProcessor, String>();

        for (Map.Entry<MessageProcessor, String> mp : messageProcessors.entrySet()) {
            MessageProcessor actualMp = mp.getKey();
            if (InterceptingChainLifecycleWrapper.class.isAssignableFrom(actualMp.getClass())) {
                String chainName = ((InterceptingChainLifecycleWrapper) actualMp).getName();
                if (!chainName.equals("(inner iterating chain) of null")) {
                    mpMap.put(actualMp, mp.getValue());
                }
            } else {
                mpMap.put(actualMp, mp.getValue());
            }
        }
        return mpMap;
    }

    /**
     * The method receives a set of paths and process them to return a map where the key is the name of a flow/sub-flow/batch
     * and the value is a list of all the paths that belong to that flow/sub-flow/batch
     *
     * @param paths set of paths
     * @return a Map where the key is the name of a flow/sub-flow, and the value is a list with all the paths that belong to the flow
     */
    public static Map<String, List<String>> buildFlowPathsMap(Set<String> paths) {
        Map<String, List<String>> flowsMap = new HashMap<String, List<String>>();
        for (String p : paths) {
            String flowName = PathParser.getFlowName(p);
            if (flowsMap.containsKey(flowName)) {
                flowsMap.get(flowName).add(p);
            } else {
                List<String> pathList = new ArrayList<String>();
                pathList.add(p);
                flowsMap.put(flowName, pathList);
            }
        }
        return flowsMap;
    }

    /**
     * The method will filter any path that belongs to a flow name listed in the variable FlowNamesToFilter
     *
     * @param paths             original set of paths
     * @param flowNamesToFilter set of name to match against the paths
     * @return a set containing the paths that passed the filter
     */
    public static Set<String> filterPaths(Set<String> paths, Set<String> flowNamesToFilter) {
        Set<String> filteredPaths = new HashSet<String>();
        for (String path : paths) {
            String flowName = PathParser.getFlowName(path);
            if (!flowNamesToFilter.contains(flowName)) {
                filteredPaths.add(path);
            }
        }
        return filteredPaths;
    }

    public static Map<String, List<String>> buildApplicationFlowPathMap(Set<String> flows, Set<String> subFlows, Set<String> batches) {
        Map<String, List<String>> flowsMap = new HashMap<String, List<String>>();

        flowsMap.putAll(buildFlowPathsMap(flows));
        flowsMap.putAll(buildFlowPathsMap(subFlows));
        flowsMap.putAll(buildFlowPathsMap(batches));

        return flowsMap;
    }

    public static Set<String> filterIgnoredFlows(Set<String> paths, Set<String> flowsToIgnore) {
        if (null != flowsToIgnore) {
            return filterPaths(paths, flowsToIgnore);
        }
        return paths;
    }
}
