/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage.report;

import org.mule.munit.plugins.coverage.path.PathBuilder;
import org.mule.munit.plugins.coverage.report.model.MuleFlow;
import org.mule.munit.plugins.coverage.report.model.MuleResource;
import java.util.*;

public class CoverageReportBuilder {

    private int coveredMpCount;
    private int totalMpCount;

    private Set<String> flowsToIgnore = new HashSet<String>();
    private Set<String> coveredPaths;
    private String applicationResources;
    private Set<String> appFlowPaths;
    private Set<String> appSubFlowPaths;
    private Set<String> appBatchPaths;
    private FileFlowMap filesMap = new FileFlowMap();
    private Map<String, List<String>> flowsMap;
    private Map<String, List<String>> flowsCoveredPaths;
    private ApplicationCoverageReport report;

    public CoverageReportBuilder() {
        reset();
    }

    public CoverageReportBuilder setApplicationResources(String applicationResources) {
        this.applicationResources = applicationResources;
        return this;
    }

    public CoverageReportBuilder setFlowsToIgnore(Set<String> flowsToIgnore) {
        this.flowsToIgnore = flowsToIgnore;
        return this;
    }

    public CoverageReportBuilder setCoveredPaths(Set<String> coveredPaths) {
        this.coveredPaths = filterIgnoredFlows(coveredPaths);
        return this;
    }

    public CoverageReportBuilder setAppFlowPaths(Set<String> appFlowPaths) {
        this.appFlowPaths = filterIgnoredFlows(appFlowPaths);
        return this;
    }

    public CoverageReportBuilder setAppSubFlowPaths(Set<String> appSubFlowPaths) {
        this.appSubFlowPaths = filterIgnoredFlows(appSubFlowPaths);
        return this;
    }

    public CoverageReportBuilder setAppBatchPaths(Set<String> appBatchPaths) {
        this.appBatchPaths = filterIgnoredFlows(appBatchPaths);
        return this;
    }

    private Set<String> filterIgnoredFlows(Set<String> paths) {
        return PathBuilder.filterIgnoredFlows(paths, flowsToIgnore);
    }

    public ApplicationCoverageReport build() {
        if(report != null) {
            return report;
        }
        report = new ApplicationCoverageReport();
        filesMap.build(applicationResources, appFlowPaths, appSubFlowPaths, appBatchPaths);
        flowsMap = PathBuilder.buildApplicationFlowPathMap(appFlowPaths, appSubFlowPaths, appBatchPaths);
        flowsCoveredPaths = PathBuilder.buildFlowPathsMap(coveredPaths);
        addMuleResourcesAndCount();
        report.setCoverage(((double)coveredMpCount*100)/totalMpCount);
        report.setResourcesWeight();

        return report;
    }

    private void addMuleResourcesAndCount() {
        for (String fileName : filesMap.getFiles()) {
            MuleResource muleResource = buildMuleResource(fileName);
            report.getResources().add(muleResource);
        }
    }

    private MuleResource buildMuleResource(String fileName) {
        MuleResource muleResource = new MuleResource(fileName);
        List<String> flowsFromFile = filesMap.get(fileName);
        addNotIgnoredFlowsToMuleResource(flowsFromFile, muleResource);
        return muleResource;
    }

    private void addNotIgnoredFlowsToMuleResource(List<String> flowsFromFile, MuleResource muleResource) {
        for (String flowName : flowsFromFile) {
            if (!flowsToIgnore.contains(flowName)) {
                MuleFlow muleFlow = new MuleFlow(flowName);
                addFlowIfPresentInFlowsMap(muleFlow);
                addFlowToCoveredPathsIfPresentInCoveredFlowsMap(muleFlow);
                muleResource.getFlows().add(muleFlow);
            }
        }
    }

    private void addFlowToCoveredPathsIfPresentInCoveredFlowsMap(MuleFlow muleFlow) {
        String flowName = muleFlow.getName();
        if (flowsCoveredPaths.containsKey(flowName)) {
            muleFlow.getCoveredPaths().addAll(flowsCoveredPaths.get(flowName));
            coveredMpCount += flowsCoveredPaths.get(flowName).size();
        }
    }

    private void addFlowIfPresentInFlowsMap(MuleFlow muleFlow) {
        String flowName = muleFlow.getName();
        if (flowsMap.containsKey(flowName)) {
            muleFlow.getPaths().addAll(flowsMap.get(flowName));
            totalMpCount += flowsMap.get(flowName).size();
        }
    }

    public void reset() {
        totalMpCount = 0;
        coveredMpCount = 0;
        report = null;
    }
}
