package org.mule.munit.plugins.coverage.path;

import org.apache.commons.lang.StringUtils;
import org.mule.api.processor.MessageProcessor;

import java.util.*;

public class PathFormatter {
    private static final String PATH_SEPARATOR = "/";

    public static Map reformatSubFlowPaths(Map paths) {
        reformatSubFlowsPaths(paths);
        reformatFlowRefToSubFlowPaths(paths, true);
        return paths;
    }

    /**
     * We need to reformat the flow-ref paths to subFlows.
     * The path will expand and show each MP in the subFlow.
     * Instead of that we just register the unique part of the the flowRef to a subFlow.
     * <p/>
     * If we get this:
     * /flowName/processors/index/subFlowName/subprocessors/index
     * <p/>
     * We only register this:
     * /flowName/processors/index/subFlowName
     *
     * @param paths
     */
    private static void reformatFlowRefToSubFlowPaths(Map<MessageProcessor, String> paths, boolean subFlowPaths) {
        for (Map.Entry<MessageProcessor, String> entry : paths.entrySet()) {
            String path = entry.getValue();
            List<String> segments = new ArrayList(Arrays.asList(PathParser.getTokens(path)));

            if (PathParser.containsSubProcessorToken(segments) && !subFlowPaths) {
                String realPath = StringUtils.join(segments.subList(0, PathParser.getSubProcessorTokenIndex(segments) - 1), PATH_SEPARATOR);
                entry.setValue(realPath);
            }

            if (PathParser.containsSubProcessorToken(segments) && subFlowPaths) {
                int idxOfSecond = PathParser.getSubProcessorsSecondAppearanceIdx(segments);
                if (idxOfSecond != -1) {
                    String realPath = StringUtils.join(segments.subList(0, idxOfSecond - 1), PATH_SEPARATOR);
                    entry.setValue(realPath);
                }
            }
        }
    }

    /**
     * Remove the first 2 tokens from the path as it add no value for registering purposes.
     *
     * @param filteredSubFlowPaths
     */
    private static void reformatSubFlowsPaths(Map<MessageProcessor, String> filteredSubFlowPaths) {
        for (Map.Entry<MessageProcessor, String> entry : filteredSubFlowPaths.entrySet()) {
            String[] tokens = PathParser.getTokens(entry.getValue());
            String[] segments = Arrays.copyOfRange(tokens, 2, tokens.length);
            String realPath = "/" + StringUtils.join(segments, PATH_SEPARATOR);
            entry.setValue(realPath);
        }
    }

    public static Map reformatBatchPaths(Map paths) {
        reformatFlowRefToSubFlowPaths(paths, false);
        return paths;
    }

    public static Map reformatFlowPaths(Map paths) {
        paths = reformatGlobalCatchPaths(paths);
        reformatFlowRefToSubFlowPaths(paths, false);
        return paths;
    }

    /**
     * Reformat the paths of the global exception strategies to take out the name of the flow.
     * Otherwise they are reported as different things (paths) when in the app is the same code.
     * In this way, we make them unique.
     * We need to make sure to count them as one in the reporter.
     * <p/>
     * This kind of paths have the following look:
     * /[flow_name]/[global_exception_strategy_id]/es/0
     *
     * @param messageProcessors
     * @return list of the mp where the paths of the global catch do not contain the flow name
     */
    private static Map<MessageProcessor, String> reformatGlobalCatchPaths(Map<MessageProcessor, String> messageProcessors) {
        Map<MessageProcessor, String> mpMap = new HashMap<MessageProcessor, String>();
        for (Map.Entry<MessageProcessor, String> mp : messageProcessors.entrySet()) {
            List<String> segments = Arrays.asList(PathParser.getTokens(mp.getValue()));
            if (PathParser.isGlobalCatchExceptionStrategy(segments)) {
                String reformattedPath = buildReformattedPathOfGlobalCatchExceptionStrategy(segments);
                mpMap.put(mp.getKey(), reformattedPath);
            } else {
                mpMap.put(mp.getKey(), mp.getValue());
            }
        }
        return mpMap;
    }

    public static String buildReformattedPathOfGlobalCatchExceptionStrategy(List<String> segments) {
        return PATH_SEPARATOR + StringUtils.join(segments.subList(2, segments.size()), PATH_SEPARATOR);
    }

    public static String getFormattedFlowName(String flowName) {
        return flowName.replaceAll(PathParser.SLASHES_THAT_ARE_NOT_PRECEDED_BY_BACKSLASH, "\\\\/");
    }
}
