/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.munit.plugins.coverage.path.PathParser;

import java.util.*;

public class CoverageCalculator {
    private transient Log log = LogFactory.getLog(this.getClass());

    private Set<String> coveredPaths;
    private Set<String> flows;
    private Set<String> subFlows;
    private Set<String> batches;

    private static class Count {
        int totalMp;
        int coveredMp;
    }

    public static class CoverageResult {
        private final Double coverage;
        private final Map<String, Double> containersCoverage;

        public CoverageResult(Double coverage, Map<String, Double> containersCoverage) {
            this.coverage = coverage;
            this.containersCoverage = containersCoverage;
        }

        public Double getCoverage() {
            return coverage;
        }

        public Map<String, Double> getContainersCoverage() {
            return containersCoverage;
        }
    }

    public CoverageCalculator(Set<String> coveredPaths, Set<String> flows, Set<String> subFlows, Set<String> batches) {
        this.coveredPaths = coveredPaths;

        this.flows = flows;
        this.subFlows = subFlows;
        this.batches = batches;

        //TODO: remove this after MULE-9996 gets fixed
        countParentAsCovered();
    }

    public CoverageResult calculate() {
        Map<String, Count> pathPerFlow = new HashMap<String, Count>();

        countFlowPaths(pathPerFlow);
        countSubFlowPaths(pathPerFlow);
        countBatchPaths(pathPerFlow);

        int totalPathCount = flows.size() + subFlows.size() + batches.size();
        Double contextCoverage = percentage(coveredPaths.size(), totalPathCount);

        Map<String, Double> containersCoverage = new HashMap<String, Double>();
        for (Map.Entry<String, Count> count : pathPerFlow.entrySet()) {
            containersCoverage.put(count.getKey(), percentage(count.getValue().coveredMp, count.getValue().totalMp));
        }

        log.debug("Coverage Calculations --> Covered Paths: " + coveredPaths.size() + " - " + "Total Paths: " + totalPathCount + " - Context Coverage: " + contextCoverage);
        return new CoverageResult(contextCoverage, containersCoverage);
    }

    private void countFlowPaths(Map<String, Count> pathPerFlow) {
        countPaths(pathPerFlow, flows, PathParser.PROCESSORS_TOKEN);
    }

    private void countSubFlowPaths(Map<String, Count> pathPerFlow) {
        countPaths(pathPerFlow, subFlows, PathParser.SUB_PROCESSORS_TOKEN);
    }

    private void countBatchPaths(Map<String, Count> pathPerFlow) {
        countPaths(pathPerFlow, batches, PathParser.BATCH_INPUT_TOKEN, PathParser.BATCH_PROCESS_RECORD_TOKEN, PathParser.BATCH_ON_COMPLETE_TOKEN);
    }

    private void countPaths(Map<String, Count> pathPerFLow, Set<String> paths, String... tokens) {
        List<String> tokenFlags = new ArrayList<String>();
        for(String token : tokens) {
            tokenFlags.add("/" + token);
        }
        for(String path : paths) {
            countPath(path, pathPerFLow, tokenFlags);
        }
    }

    private void countPath(String path, Map<String, Count> pathPerFlow, List<String> tokenFlags) {
        String containerName = buildContainerName(path, tokenFlags);
        if (!pathPerFlow.containsKey(containerName)) {
            pathPerFlow.put(containerName, new Count());
        }
        Count count = pathPerFlow.get(containerName);
        count.totalMp++;
        if (coveredPaths.contains(path)) {
            count.coveredMp++;
        }
    }

    private String buildContainerName(String path, List<String> tokenFlags) {
        for (String tokenFlag : tokenFlags) {
            int tokenIdx = path.indexOf(tokenFlag);
            if(tokenIdx != -1) {
                return path.substring(0, tokenIdx);
            }
        }
        return null;
    }

    private double percentage(int part, int total) {
        if (total != 0) {
            return ((double) part * 100) / total;
        }
        return 0;
    }

    private void countParentAsCovered() {
        Set<String> coveredParentPaths = new HashSet<String>();
        for (String cp : coveredPaths) {
            String parentPath = PathParser.getParentPath(cp);
            if (StringUtils.isNotBlank(parentPath)) {
                if (flows.contains(parentPath) || subFlows.contains(parentPath) || batches.contains(parentPath)) {
                    coveredParentPaths.add(parentPath);
                }
            }
        }
        coveredPaths.addAll(coveredParentPaths);
    }
}
