/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage.server;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketTimeoutException;

import org.apache.commons.lang.Validate;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.mule.munit.plugins.coverage.report.SingleRunReport;

public class MunitCoverageServer implements Runnable, ReportAccumulator {

    // 30 mins in millis
    private static final int SOCKET_TIMEOUT_MILLIS = 1800000;

    private transient Log log = LogFactory.getLog(this.getClass());

    private int port;
    private boolean running = false;
    private boolean keepRunning = true;

    private ReportAccumulator reportAccumulator;

    public MunitCoverageServer(Integer port, ReportAccumulator reportAccumulator) {
        Validate.notNull(port, "The port can not be null");
        Validate.notNull(reportAccumulator, "The report accumulator can not be null");

        this.port = port;
        this.reportAccumulator = reportAccumulator;
    }

    public synchronized Thread launch() {
        if(running){
            throw new RuntimeException("The Coverage server is already running it can not be started again.");
        }

        Thread thread = new Thread(this);
        thread.start();
        return thread;
    }

    public synchronized void shutdown() {
        this.running = false;
        this.keepRunning = false;
    }

    public void run() {
        running = true;
        keepRunning = true;
        ServerSocket providerSocket = null;

        try {
            providerSocket = new ServerSocket(port, 10);
            log.debug("Coverage server started listening in port [" + port + "]...");
            providerSocket.setSoTimeout(SOCKET_TIMEOUT_MILLIS);

            do {
                log.info("Waiting for coverage connection ");
                Socket connection = providerSocket.accept();
                log.info("Coverage connection received from " + connection.getInetAddress().getHostName() + " - " + keepRunning);

                CoverageMessageParser parser = new CoverageMessageParser(new ObjectInputStream(connection.getInputStream()), this);
                new Thread(parser).run();
            } while (keepRunning);

        } catch (SocketTimeoutException timeoutException) {
            log.debug("Coverage server time out");
            if (keepRunning) {
                log.error("Coverage connection timeout after " + String.valueOf(SOCKET_TIMEOUT_MILLIS) + " milliseconds");
            }
        } catch (IOException ioException) {
            log.error("Failed to start Coverage server in port " + port);
        } finally {
            try {
                log.debug("Shutting down coverage server running in port [" + port + "]...");

                if (null != providerSocket) {
                    providerSocket.close();
                }

                running = false;
                keepRunning = false;
                log.debug("Coverage server shutdown");
            } catch (IOException ioException) {
                log.debug("Coverage server error during shutdown.");
            }
        }

    }

    public int getPort() {
        return this.port;
    }

    public boolean isRunning() {
        return running;
    }

    @Override
    public synchronized void accumulateReport(SingleRunReport report) {
        reportAccumulator.accumulateReport(report);
    }

}
