/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage;

import static java.lang.Integer.valueOf;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.apache.commons.lang3.math.NumberUtils.isDigits;

import java.io.File;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.mule.munit.plugins.coverage.core.CoverageModule;
import org.mule.runtime.api.config.custom.CustomizationService;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.module.deployment.api.DeploymentListener;
import org.mule.runtime.module.deployment.api.DeploymentService;
import org.mule.runtime.module.deployment.api.DeploymentServiceAware;

import com.mulesoft.mule.runtime.module.plugin.api.MulePlugin;

/**
 * Entry point and hook of coverage as {@link MulePlugin}
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class CoverageServerPlugin implements MulePlugin, DeploymentServiceAware, DeploymentListener {

  public static final String COVERAGE_PORT_PROPERTY = "coverage.port";
  public static final String COVERAGE_SEND_ALL_LOCATIONS_PROPERTY = "coverage.send.all.locations";

  private transient Log logger = LogFactory.getLog(this.getClass());

  protected final String port;
  protected CoverageModule coverageModule;
  protected DeploymentService deploymentService;

  public CoverageServerPlugin() {
    port = System.getProperty(COVERAGE_PORT_PROPERTY, null);
  }

  @Override
  public void dispose() {
    // TODO null all properties (asume to be collected)
    // coverageModule = null;
    // deploymentService = null;
  }

  @Override
  public void initialise() throws InitialisationException {
    logger.info("Initialising coverage plugin...");
  }

  @Override
  public void start() throws MuleException {
    logger.debug("Starting coverage plugin...");
  }

  @Override
  public void stop() throws MuleException {
    logger.debug("Stopping coverage plugin...");
  }

  @Override
  public void setWorkingDirectory(File file) {}

  @Override
  public boolean isDisabledOnEnvironment() {
    return false;
  }

  @Override
  public void setDeploymentService(DeploymentService deploymentService) {
    this.deploymentService = deploymentService;
    if (this.deploymentService != null) {
      this.deploymentService.addDeploymentListener(this);
    }
  }

  @Override
  public void onUndeploymentStart(String artifactName) {
    if (shouldEnableCoverageServerPlugin()) {
      if (shouldSendAllLocations()) {
        logger.debug("Sending all locations report...");
        coverageModule.sendAllLocationReport(valueOf(port));
      } else {
        logger.debug("Sending covered locations report...");
        coverageModule.sendCoveredLocationsReport(valueOf(port));
      }
      logger.debug("Coverage location report sent");
    }
  }

  @Override
  public void onMuleContextCreated(String artifactName, MuleContext context, CustomizationService customizationService) {
    if (shouldEnableCoverageServerPlugin() && coverageModule == null) {
      logger.debug("Coverage Server plugin enabled. Registering interceptor");
      coverageModule = new CoverageModule(context.getProcessorInterceptorManager());
    }
  }

  @Override
  public void onMuleContextInitialised(String artifactName, MuleContext context) {
    if (coverageModule != null) {
      coverageModule.setComponentLocator(context.getConfigurationComponentLocator());
    }
  }

  private boolean shouldEnableCoverageServerPlugin() {
    return isNotBlank(port) && isDigits(port);
  }

  private boolean shouldSendAllLocations() {
    return Boolean.valueOf(System.getProperty(COVERAGE_SEND_ALL_LOCATIONS_PROPERTY, "false"));
  }
}
