/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage.core;


import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static org.mule.munit.plugins.coverage.core.interception.ComponentLocationFilter.isRepresentativeForCoverage;
import static org.mule.munit.plugins.coverage.core.model.CoverageComponentLocation.fromComponentLocation;

import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.mule.munit.plugins.coverage.core.interception.CoverageProcessorInterceptor;
import org.mule.munit.plugins.coverage.core.interception.CoverageProcessorInterceptorFactory;
import org.mule.munit.plugins.coverage.core.model.CoverageComponentLocation;
import org.mule.munit.plugins.coverage.server.CoverageLocations;
import org.mule.munit.plugins.coverage.server.CoverageServerClient;
import org.mule.runtime.api.component.location.ConfigurationComponentLocator;
import org.mule.runtime.core.api.interception.ProcessorInterceptorProvider;

/**
 * It hocks all the coverage components together, and provide access to the {@link LocationAccumulator}
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class CoverageModule {

  private transient Log logger = LogFactory.getLog(this.getClass());

  protected ConfigurationComponentLocator componentLocator;

  protected ProcessorInterceptorProvider processorInterceptorProvider;

  protected LocationAccumulator locationAccumulator;

  public CoverageModule(ProcessorInterceptorProvider processorInterceptorProvider) {
    checkNotNull(processorInterceptorProvider, "The interceptor provider must not be null");

    this.processorInterceptorProvider = processorInterceptorProvider;
    this.locationAccumulator = new LocationAccumulator();

    registerInterceptorFactory();
  }

  public void setComponentLocator(ConfigurationComponentLocator componentLocator) {
    checkNotNull(componentLocator, "The component location provider must not be null");
    this.componentLocator = componentLocator;
  }

  public LocationAccumulator getLocationAccumulator() {
    return locationAccumulator;
  }

  public void sendCoveredLocationsReport(Integer port) {
    CoverageLocations report = new CoverageLocations(locationAccumulator.getCoverageLocations());
    sendReport(port, report);
  }

  public void sendAllLocationReport(Integer port) {
    checkArgument(componentLocator != null, "The componentLocator hasn't been provided.");

    Set<CoverageComponentLocation> componentLocations =
        componentLocator.findAll().stream()
            .filter(cl -> isRepresentativeForCoverage(cl.getLocation()))
            .map(cl -> fromComponentLocation(cl.getLocation()))
            .collect(Collectors.toSet());

    CoverageLocations report = new CoverageLocations(componentLocations);
    report.setAllLocations(true);
    sendReport(port, report);
  }

  private void registerInterceptorFactory() {
    logger.debug("Adding " + CoverageProcessorInterceptor.class.getName() + " to processor interceptor manager");
    processorInterceptorProvider.addInterceptorFactory(new CoverageProcessorInterceptorFactory(locationAccumulator));
  }

  protected void sendReport(Integer port, CoverageLocations report) {
    checkNotNull(port, "The port must not be null");
    checkArgument(port > 0, "The port must be a positive number");

    new CoverageServerClient(port).sendCoveredLocations(report);
  }
}
