/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage.core.interception;

import static com.google.common.base.Preconditions.checkNotNull;

import java.util.Map;
import java.util.concurrent.CompletableFuture;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.mule.munit.plugins.coverage.core.LocationAccumulator;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.interception.InterceptionAction;
import org.mule.runtime.api.interception.InterceptionEvent;
import org.mule.runtime.api.interception.ProcessorInterceptor;
import org.mule.runtime.api.interception.ProcessorParameterValue;

/**
 * It's the Coverage Interceptor.
 * <p>
 * It register the call of all the processors in the application.
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class CoverageProcessorInterceptor implements ProcessorInterceptor {

  private transient Log logger = LogFactory.getLog(this.getClass());

  private LocationAccumulator locationAccumulator;

  public CoverageProcessorInterceptor(LocationAccumulator locationAccumulator) {
    checkNotNull(locationAccumulator, "The reporter must not be null");
    this.locationAccumulator = locationAccumulator;
  }

  @Override
  public CompletableFuture<InterceptionEvent> around(ComponentLocation location,
                                                     Map<String, ProcessorParameterValue> parameters,
                                                     InterceptionEvent event, InterceptionAction action) {

    logger.debug("About to register execution of: " + locationToStringLog(location));
    locationAccumulator.addCoveredLocation(location);
    return action.proceed();

  }

  public LocationAccumulator getLocationAccumulator() {
    return locationAccumulator;
  }

  private String locationToStringLog(ComponentLocation location) {
    return location.getComponentIdentifier().getIdentifier()
        + " in "
        + location.getFileName().orElse(" ? ")
        + "[line: " + location.getLineInFile().orElse(-1) + "].";
  }

}
