/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage.core.model;

import static java.util.Optional.ofNullable;
import static org.mule.munit.plugins.coverage.core.model.CoverageTypedComponentIdentifier.fromTypedComponentIdentifier;

import java.io.Serializable;
import java.util.Optional;

import org.mule.runtime.api.component.TypedComponentIdentifier;
import org.mule.runtime.api.component.location.LocationPart;

/**
 * A location part represent an specific location of a component within another component.
 * 
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class CoverageLocationPart implements LocationPart, Serializable {

  private static final long serialVersionUID = 5757545892752260058L;

  private String partPath;
  private String fileName;
  private Integer lineInFile;
  private CoverageTypedComponentIdentifier partIdentifier;

  /**
   * @param partPath the path of this part
   * @param partIdentifier the component identifier of the part if it's not a synthetic part
   * @param fileName the file name in which the component was defined
   * @param lineInFile the line number in which the component was defined
   */
  public CoverageLocationPart(String partPath, Optional<CoverageTypedComponentIdentifier> partIdentifier,
                              Optional<String> fileName,
                              Optional<Integer> lineInFile) {
    this.partPath = partPath;
    this.partIdentifier = partIdentifier.orElse(null);
    fileName.ifPresent(configFileName -> this.fileName = configFileName);
    lineInFile.ifPresent(line -> this.lineInFile = line);
  }

  /**
   * @return the string representation of the part
   */
  @Override
  public String getPartPath() {
    return partPath;
  }

  /**
   * @return if it's a synthetic part this is null, if not then it's the identifier of the configuration element.
   */
  @Override
  public Optional<TypedComponentIdentifier> getPartIdentifier() {
    return ofNullable(partIdentifier);
  }

  @Override
  public Optional<String> getFileName() {
    return ofNullable(fileName);
  }

  @Override
  public Optional<Integer> getLineInFile() {
    return ofNullable(lineInFile);
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    CoverageLocationPart that = (CoverageLocationPart) o;

    if (!getPartPath().equals(that.getPartPath())) {
      return false;
    }
    if (getPartIdentifier() != null ? !getPartIdentifier().equals(that.getPartIdentifier())
        : that.getPartIdentifier() != null) {
      return false;
    }
    if (getFileName() != null ? !getFileName().equals(that.getFileName()) : that.getFileName() != null) {
      return false;
    }
    return getLineInFile() != null ? getLineInFile().equals(that.getLineInFile()) : that.getLineInFile() == null;
  }

  @Override
  public int hashCode() {
    int result = getPartPath().hashCode();
    result = 31 * result + (getPartIdentifier() != null ? getPartIdentifier().hashCode() : 0);
    result = 31 * result + (getFileName() != null ? getFileName().hashCode() : 0);
    result = 31 * result + (getLineInFile() != null ? getLineInFile().hashCode() : 0);
    return result;
  }

  @Override
  public String toString() {
    return "CoverageLocationPart{" +
        "partPath='" + partPath + '\'' +
        ", partIdentifier=" + partIdentifier +
        ", fileName='" + fileName + '\'' +
        ", lineInFile=" + lineInFile +
        '}';
  }


  public static CoverageLocationPart fromLocationPart(LocationPart lp) {
    return new CoverageLocationPart(lp.getPartPath(),
                                    fromTypedComponentIdentifier(lp.getPartIdentifier()),
                                    lp.getFileName(),
                                    lp.getLineInFile());
  }



}


