/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage.report.printer;

import static com.google.common.base.Preconditions.checkNotNull;

import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import org.mule.munit.common.util.IOUtils;
import org.mule.munit.plugins.coverage.report.model.ApplicationCoverageReport;
import org.mule.munit.plugins.coverage.report.model.CoverageLimits;
import org.mule.munit.plugins.coverage.report.model.MuleResource;

import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;

/**
 * <p>
 * Generates HTML Coverage Report.
 * </p>
 * 
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class HtmlPrinter implements CoverageReportPrinter {

  public static final String PRINTER_NAME = "HTML";

  private static final String ASSETS_FOLDER = "assets";
  private static final String STYLES_FOLDER = "styles";
  private static final String SCRIPTS_FOLDER = "js";

  private static final String BASE_TEMPLATE_FOLDER = "/templates";
  private static final String SUMMARY_TEMPLATE_FILE_NAME = "summary.ftl";
  private static final String RESOURCE_TEMPLATE_FILE_NAME = "resource.ftl";

  private static final String MULESOFT_STYLES = "mulesoft-styles.css";
  private static final String TSORTER_STYLES = "tsorter.css";
  private static final String MULESOFT_ICON = "icons/004_logo.svg";
  private static final String MULESOFT_ICON_TTF = "fonts/muleicons.ttf";
  private static final String TSORTER_SCRIPT = "tsorter.min.js";

  private final File reportDirectory;
  private CoverageLimits coverageLimits;
  private ReportFileWriter reportFileWriter = new ReportFileWriter();
  private Configuration freemarkerConfig;

  public HtmlPrinter(CoverageLimits coverageLimits, File reportDirectory) {
    checkNotNull(coverageLimits, "Coverage limits cannot be null");
    checkNotNull(reportDirectory, "Report directory cannot be null");
    this.coverageLimits = coverageLimits;
    this.reportDirectory = reportDirectory;
    this.freemarkerConfig = new Configuration(Configuration.VERSION_2_3_23);
    this.freemarkerConfig.setClassForTemplateLoading(getClass(), BASE_TEMPLATE_FOLDER);
  }

  public String getPrinterName() {
    return PRINTER_NAME;
  }

  public void printReport(ApplicationCoverageReport report) {
    try {
      printSummary(report);
      printResources(report);
      addAssets();
    } catch (IOException e) {
      e.printStackTrace();
    }
  }

  public void setReportFileWriter(ReportFileWriter reportFileWriter) {
    this.reportFileWriter = reportFileWriter;
  }

  private void printSummary(ApplicationCoverageReport report) throws IOException {
    StringWriter writer = null;
    try {
      Template summaryTemplate = getTemplate(SUMMARY_TEMPLATE_FILE_NAME);
      Map<String, Object> data = new HashMap<>();
      data.put("applicationFlowCount", report.getApplicationFlowCount());
      data.put("applicationCoverage", report.getCoverage());
      data.put("applicationResourceLines", report.getResources());
      data.put("coverageLimits", coverageLimits);

      writer = new StringWriter();
      summaryTemplate.process(data, writer);
      reportFileWriter.writeToFile(new File(reportDirectory, "summary.html"), writer.getBuffer().toString());
    } catch (TemplateException e) {
      e.printStackTrace();
    } finally {
      IOUtils.closeQuietly(writer);
    }
  }

  private void printResources(ApplicationCoverageReport report) throws IOException {
    StringWriter writer = null;
    for (MuleResource mr : report.getResources()) {
      try {
        Template resourceTemplate = getTemplate(RESOURCE_TEMPLATE_FILE_NAME);

        Map<String, Object> data = new HashMap<>();
        data.put("resourceName", mr.getName());
        data.put("resourceMpCovered", mr.getCoveredMessageProcessorCount());
        data.put("resourceMpCount", mr.getMessageProcessorCount());
        data.put("resourceCoverage", mr.getCoverage());
        data.put("flowLines", mr.getFlows());
        data.put("coverageLimits", coverageLimits);

        writer = new StringWriter();
        resourceTemplate.process(data, writer);
        reportFileWriter.writeToFile(new File(reportDirectory, reportFileWriter.getHtmlFileName(mr.getName())),
                                     writer.getBuffer().toString());
      } catch (TemplateException e) {
        e.printStackTrace();
      } finally {
        IOUtils.closeQuietly(writer);
      }
    }
  }

  private Template getTemplate(String resourceTemplateFileName) throws IOException {
    return freemarkerConfig.getTemplate(resourceTemplateFileName, Locale.US);
  }

  private void addAssets() {
    addStyles();
    addScripts();
  }

  private void addStyles() {
    try {
      Path stylesPath = reportDirectory.toPath().resolve(ASSETS_FOLDER).resolve(STYLES_FOLDER);
      reportFileWriter.copyFile(MULESOFT_STYLES, stylesPath.resolve(MULESOFT_STYLES).toFile());
      reportFileWriter.copyFile(MULESOFT_ICON, stylesPath.resolve(MULESOFT_ICON).toFile());
      reportFileWriter.copyFile(MULESOFT_ICON_TTF, stylesPath.resolve(MULESOFT_ICON_TTF).toFile());
      reportFileWriter.copyFile(TSORTER_STYLES, stylesPath.resolve(TSORTER_STYLES).toFile());
    } catch (IOException e) {
      e.printStackTrace();
    }
  }

  private void addScripts() {
    try {
      Path scriptsPath = reportDirectory.toPath().resolve(ASSETS_FOLDER).resolve(SCRIPTS_FOLDER);
      reportFileWriter.copyFile(TSORTER_SCRIPT, scriptsPath.resolve(TSORTER_SCRIPT).toFile());
    } catch (IOException e) {
      e.printStackTrace();
    }
  }
}
