/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage;

import static java.lang.Integer.valueOf;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.apache.commons.lang3.math.NumberUtils.isDigits;
import static org.mule.munit.plugins.coverage.core.interception.CoverageProcessorInterceptorFactory.MUNIT_COVERAGE_PROCESSOR_INTERCEPTOR_FACTORY_ID;

import java.io.File;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.mule.munit.plugins.coverage.core.CoverageModule;
import org.mule.munit.plugins.coverage.core.interception.CoverageProcessorInterceptorFactory;
import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.component.location.ConfigurationComponentLocator;
import org.mule.runtime.api.config.custom.CustomizationService;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.module.deployment.api.DeploymentListener;
import org.mule.runtime.module.deployment.api.DeploymentService;
import org.mule.runtime.module.deployment.api.DeploymentServiceAware;

import com.mulesoft.mule.runtime.module.plugin.api.MulePlugin;

/**
 * Entry point and hook of coverage as {@link MulePlugin}
 *
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class CoverageServerPlugin implements MulePlugin, DeploymentServiceAware, DeploymentListener {

  public static final String COVERAGE_PORT_PROPERTY = "coverage.port";

  private transient Log logger = LogFactory.getLog(this.getClass());

  protected final String port;
  protected CoverageModule coverageModule;
  protected DeploymentService deploymentService;

  public CoverageServerPlugin() {
    port = System.getProperty(COVERAGE_PORT_PROPERTY, null);
  }

  @Override
  public void dispose() {
    // TODO null all properties (asume to be collected)
    // coverageModule = null;
    // deploymentService = null;
  }

  @Override
  public void initialise() throws InitialisationException {
    logger.info("Initialising coverage plugin...");
  }

  @Override
  public void start() throws MuleException {
    logger.debug("Starting coverage plugin...");
  }

  @Override
  public void stop() throws MuleException {
    logger.debug("Stopping coverage plugin...");
  }

  @Override
  public void setWorkingDirectory(File file) {}

  @Override
  public boolean isDisabledOnEnvironment() {
    return false;
  }

  @Override
  public void setDeploymentService(DeploymentService deploymentService) {
    this.deploymentService = deploymentService;
    if (this.deploymentService != null) {
      this.deploymentService.addDeploymentListener(this);
    }
  }

  @Override
  public void onUndeploymentStart(String artifactName) {
    if (shouldEnableCoverageServerPlugin()) {
      logger.debug("Sending covered locations report...");
      coverageModule.sendCoveredLocationsReport(valueOf(port));

      logger.debug("Sending all locations report...");
      coverageModule.sendAllLocationReport(valueOf(port));

      logger.debug("Coverage location reports sent");
      coverageModule = null;
    }
  }

  @Override
  public void onArtifactCreated(String artifactName, CustomizationService customizationService) {
    if (shouldEnableCoverageServerPlugin() && coverageModule == null) {
      logger.debug("Coverage Server plugin enabled. Registering interceptor");
      coverageModule = new CoverageModule();

      CoverageProcessorInterceptorFactory interceptorFactory =
          new CoverageProcessorInterceptorFactory(coverageModule.getLocationAccumulator());
      customizationService.registerCustomServiceImpl(MUNIT_COVERAGE_PROCESSOR_INTERCEPTOR_FACTORY_ID,
                                                     interceptorFactory);
    }
  }

  @Override
  public void onArtifactInitialised(String artifactName, Registry registry) {
    if (coverageModule != null) {
      ConfigurationComponentLocator locator =
          registry.<ConfigurationComponentLocator>lookupByName(ConfigurationComponentLocator.REGISTRY_KEY).get();
      coverageModule.setComponentLocator(locator);
    }
  }

  private boolean shouldEnableCoverageServerPlugin() {
    return isNotBlank(port) && isDigits(port);
  }
}
