/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage.server;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketTimeoutException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * It listens for notifications of the coverage module reporting coverage data
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class CoverageServer implements Runnable {


  private static final int SOCKET_TIMEOUT_MILLIS = 30 * 60 * 1000;

  private transient Log log = LogFactory.getLog(this.getClass());

  private Integer port;
  private Boolean running = false;
  private Boolean keepRunning = true;

  private CoverageLocationsAccumulator coverageLocationsAccumulator;

  public CoverageServer(Integer port, CoverageLocationsAccumulator coverageLocationsAccumulator) {
    checkNotNull(port, "The port can not be null.");
    checkArgument(port > 0, "The port must be a positive number.");
    checkNotNull(coverageLocationsAccumulator, "The report accumulator must not be null.");

    this.port = port;
    this.coverageLocationsAccumulator = coverageLocationsAccumulator;
  }

  public int getPort() {
    return this.port;
  }

  public boolean isRunning() {
    return running;
  }

  public synchronized void launch() {
    if (running) {
      throw new RuntimeException("The Coverage server is already running it can not be started again.");
    }
    new Thread(this).start();
  }

  public synchronized void shutDown() {
    this.keepRunning = false;
  }

  public void run() {
    log.info("Coverage Server starting...");
    running = true;
    keepRunning = true;
    ServerSocket providerSocket = null;

    try {
      providerSocket = new ServerSocket(port, 10);
      providerSocket.setSoTimeout(SOCKET_TIMEOUT_MILLIS);
      log.info("Waiting for coverage client connection in port [" + port + "]...");
      do {
        Socket connection = providerSocket.accept();
        log.info("Coverage client connection received from " + connection.getInetAddress().getHostName() + " - " + keepRunning);

        CoverageMessageParser parser =
            new CoverageMessageParser(new ObjectInputStream(connection.getInputStream()), coverageLocationsAccumulator);
        new Thread(parser).run();
      } while (keepRunning);

    } catch (SocketTimeoutException timeoutException) {
      log.warn("Coverage Server time out");
      if (keepRunning) {
        log.error("Coverage Server connection timeout after " + String.valueOf(SOCKET_TIMEOUT_MILLIS) + " milliseconds");
      }
    } catch (IOException ioException) {
      log.error("Failed to start Coverage Server in port " + port);
    } finally {
      try {
        log.debug("Shutting down coverage server running in port [" + port + "]...");
        running = false;
        keepRunning = false;
        if (null != providerSocket) {
          providerSocket.close();
        }
        log.debug("Coverage server shutdown");
      } catch (IOException ioException) {
        log.debug("Coverage Server error during shut down.");
      }
    }
  }
}
