/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage.report.printer;

import java.util.List;
import java.util.Locale;
import java.util.Scanner;

import org.mule.munit.plugins.coverage.report.model.ApplicationCoverageReport;
import org.mule.munit.plugins.coverage.report.model.MuleFlow;
import org.mule.munit.plugins.coverage.report.model.MuleResource;

import de.vandermeer.asciitable.v2.V2_AsciiTable;
import de.vandermeer.asciitable.v2.render.V2_AsciiTableRenderer;
import de.vandermeer.asciitable.v2.render.WidthLongestLine;
import de.vandermeer.asciitable.v2.themes.V2_E_TableThemes;

/**
 * <p>
 * Prints Coverage Report through the Console.
 * </p>
 * 
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public abstract class ConsolePrinter implements CoverageReportPrinter {

  public static final String PRINTER_NAME = "Console";

  /**
   * Represents the table alignment, each letter is left, right or center
   */
  private static final char[] TABLE_ALIGNMENT = {'l', 'l', 'c', 'c', 'c'};

  public String getPrinterName() {
    return PRINTER_NAME;
  }


  public void printReport(ApplicationCoverageReport report) {
    log("===============================================================================");
    log("\t\t\t\tMUnit Coverage Report");
    log("===============================================================================");
    log("");

    V2_AsciiTableRenderer flowTableRenderer = new V2_AsciiTableRenderer();
    flowTableRenderer.setWidth(new WidthLongestLine());

    V2_AsciiTableRenderer resourceRowRenderer = new V2_AsciiTableRenderer();
    resourceRowRenderer.setWidth(new WidthLongestLine());
    resourceRowRenderer.setTheme(V2_E_TableThemes.UTF_DOUBLE.get());

    for (MuleResource mr : report.getResources()) {
      String resourceLine;
      if (mr.getCoverage() != -1) {
        resourceLine = "Resource File: " + mr.getName() + " - Containers: " + mr.getFlows().size() + " - Weight: "
            + formatDecimals(mr.getWeight()) + " -  Coverage: " + formatDecimals(mr.getCoverage()) + "%";
      } else {
        resourceLine = "Resource File: " + mr.getName() + " - Containers: " + mr.getFlows().size() + " - Weight: "
            + formatDecimals(mr.getWeight()) + " -  Coverage: N/A";
      }
      renderResourceRow(resourceRowRenderer, resourceLine);
      renderContainerTable(flowTableRenderer, mr);
    }

    log("");
    log("===============================================================================");
    log("\t\t\t\t\tSummary");
    log("===============================================================================");
    log("  * Covered Processors: " + report.getApplicationCoveredMessageProcessorCount());
    log("  * Total Processors:   " + report.getApplicationMessageProcessorCount());
    log("  * Containers:         " + report.getApplicationFlowCount());
    log("  * Resources:          " + report.getResources().size());
    log("===============================================================================");
    if (report.getApplicationFlowCount() > 0) {
      log("\t\t\t\t** Application Coverage: " + formatDecimals(report.getCoverage()) + "% **");
    } else {
      log("\t\t\t\t** Application Coverage: N/A **");
    }
    log("===============================================================================");

  }

  protected abstract void log(String message);

  private void renderContainerTable(V2_AsciiTableRenderer flowTableRenderer, MuleResource mr) {
    Scanner flowCoverageTableScanner = getContainerCoverageTable(mr.getFlows(), flowTableRenderer);
    while (flowCoverageTableScanner.hasNextLine()) {
      log(flowCoverageTableScanner.nextLine());
    }
  }

  private void renderResourceRow(V2_AsciiTableRenderer resourceRowRenderer, String resourceLine) {
    V2_AsciiTable resourceRow = new V2_AsciiTable();
    resourceRow.addRule();
    resourceRow.addRow(resourceLine);
    resourceRow.addRule();

    Scanner resourceRowScanner = new Scanner(resourceRowRenderer.render(resourceRow).toString());
    while (resourceRowScanner.hasNextLine()) {
      log(resourceRowScanner.nextLine());
    }
  }

  private Scanner getContainerCoverageTable(List<MuleFlow> flows, V2_AsciiTableRenderer renderer) {
    V2_AsciiTable containerTable = new V2_AsciiTable();
    containerTable.addRule();
    containerTable.addRow("Name", "Type", "Covered (P)", "Total (P)", "Coverage").setAlignment(TABLE_ALIGNMENT);
    containerTable.addRule();
    for (MuleFlow mf : flows) {
      containerTable.addRow(mf.getName(), mf.getPrintableType(), mf.getCoveredLocations().size(), mf.getLocations().size(),
                            formatDecimals(mf.getCoverage()) + "%")
          .setAlignment(TABLE_ALIGNMENT);
    }
    containerTable.addRule();
    return new Scanner(renderer.render(containerTable).toString());
  }

  private String formatDecimals(Double number) {
    return String.format(Locale.ENGLISH, "%.2f", number);
  }

}
