/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage.report.printer;

import com.google.gson.GsonBuilder;
import org.mule.munit.plugins.coverage.report.model.ApplicationCoverageReport;
import org.mule.munit.plugins.coverage.report.model.CoverageLimits;
import org.mule.munit.plugins.coverage.report.model.MuleFlow;
import org.mule.munit.plugins.coverage.report.model.MuleResource;
import org.mule.munit.plugins.coverage.report.printer.model.CoverageReportModel;
import org.mule.munit.plugins.coverage.report.printer.model.MuleFlowModel;
import org.mule.munit.plugins.coverage.report.printer.model.MuleResourceModel;

import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.List;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.nio.charset.StandardCharsets.UTF_8;

/**
 * <p>
 * Generates JSON file Coverage Report.
 * </p>
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class JsonPrinter implements CoverageReportPrinter {

  public static final String PRINTER_NAME = "Json";
  private final File reportFile;
  private CoverageLimits configuration;

  public JsonPrinter(CoverageLimits coverageLimits, File reportFile) {
    checkNotNull(reportFile, "The report path cannot be null");
    checkNotNull(coverageLimits, "The coverage limits cannot be null");
    this.configuration = coverageLimits;
    this.reportFile = reportFile;
  }

  public String getPrinterName() {
    return PRINTER_NAME;
  }

  public void printReport(ApplicationCoverageReport report) {
    try {
      CoverageReportModel coverageReportModel = generateReportModel(report);
      reportFile.createNewFile();

      Writer writer = new OutputStreamWriter(new FileOutputStream(reportFile), UTF_8);
      new GsonBuilder().setPrettyPrinting().create().toJson(coverageReportModel, writer);
      writer.close();
    } catch (Exception e) {
      e.printStackTrace();
    }
  }

  private CoverageReportModel generateReportModel(ApplicationCoverageReport report) {
    CoverageReportModel coverageReportModel = new CoverageReportModel(report, configuration);
    List<MuleResourceModel> files = new ArrayList<>();
    for (MuleResource resource : report.getResources()) {
      MuleResourceModel resourceModel = new MuleResourceModel(resource);
      List<MuleFlowModel> flows = new ArrayList<>();
      for (MuleFlow flow : resource.getFlows()) {
        flows.add(new MuleFlowModel(flow));
      }
      resourceModel.setFlows(flows);
      files.add(resourceModel);
    }
    coverageReportModel.setFiles(files);
    return coverageReportModel;
  }
}
