/*
 * Copyright (c) 2017 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage.server;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;

import java.io.IOException;
import java.io.ObjectOutputStream;
import java.net.Socket;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.gson.Gson;


/**
 * Sends a {@link CoverageLocations} through a socket
 *
 * @author Mulesoft Inc.
 * @since 1.0.0
 */
public class CoverageServerClient {

  private static final String DEFAULT_HOST = "localhost";

  private transient Log logger = LogFactory.getLog(this.getClass());

  private Integer serverPort;

  public CoverageServerClient(Integer serverPort) {
    checkNotNull(serverPort, "The server port must not be null.");
    checkArgument(serverPort > 0, "The server port must be a positive number.");
    this.serverPort = serverPort;
  }

  public void sendCoveredLocations(CoverageLocations report) {
    checkNotNull(report, "The report must not be null.");
    send(reportToJson(report));
  }

  protected String reportToJson(CoverageLocations report) {
    Gson gson = new Gson();
    return gson.toJson(report);
  }

  protected void send(String report) {
    logger.info("Sending covered locations report...");
    Socket requestSocket = null;
    ObjectOutputStream out = null;

    try {
      requestSocket = new Socket(DEFAULT_HOST, serverPort);
      logger.info("Connected to " + DEFAULT_HOST + " in port " + serverPort);

      out = new ObjectOutputStream(requestSocket.getOutputStream());

      out.writeObject(report);
      out.flush();
      logger.debug("Covered locations report sent");
    } catch (IOException ioException) {
      logger.error("Fail to send covered locations report");
      ioException.printStackTrace();
    } finally {
      logger.debug("Shutting down " + this.getClass().getName());
      try {
        if (out != null) {
          out.close();
        }
        if (requestSocket != null) {
          requestSocket.close();
        }
      } catch (IOException ioException) {
        ioException.printStackTrace();
      }
    }
  }

}

