/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage.extensions;

import org.mule.MuleCoreExtension;
import org.mule.api.MuleException;
import org.mule.api.lifecycle.InitialisationException;
import org.mule.module.launcher.DeploymentService;
import org.mule.module.launcher.DeploymentServiceAware;
import org.mule.module.launcher.PluginClassLoaderManager;
import org.mule.module.launcher.PluginClassLoaderManagerAware;
import org.mule.util.FileUtils;

import com.mulesoft.mule.plugin.ManagedMulePlugin;
import com.mulesoft.mule.plugin.classloader.FileSystemPluginClassLoaderFactory;
import com.mulesoft.mule.plugin.classloader.FilteringPluginClassLoader;
import com.mulesoft.mule.plugin.classloader.PluginClassLoaderFilter;
import com.mulesoft.mule.plugin.discoverer.FileSystemPluginDiscoverer;
import com.mulesoft.mule.plugin.factory.MulePluginConfigurerFactory;
import com.mulesoft.mule.plugin.factory.MulePluginFactory;
import com.mulesoft.mule.plugin.factory.MulePluginPropertyDiscovererFactory;
import com.mulesoft.mule.plugin.factory.PluginFactory;
import com.mulesoft.mule.plugin.manager.MulePluginManagerFactory;
import com.mulesoft.mule.plugin.manager.PluginManager;
import com.mulesoft.mule.plugin.manager.PluginRegistrationListener;
import com.mulesoft.mule.plugin.processor.CompositePluginProcessor;
import com.mulesoft.mule.plugin.processor.PluginProcessor;
import com.mulesoft.mule.plugin.processor.deployment.DeploymentListenerProvider;
import com.mulesoft.mule.plugin.processor.deployment.DeploymentListenerProviderProcessor;
import com.mulesoft.mule.plugin.processor.deployment.DeploymentServiceAwareProcessor;

import java.io.File;
import java.io.IOException;

/**
 *
 */
public class MunitPluginCoreExtension implements MuleCoreExtension, DeploymentServiceAware, PluginClassLoaderManagerAware, PluginRegistrationListener
{
    public static final String PLUGINS_DIR = "mule-plugins";

    private PluginManager pluginManager;

    private DeploymentService deploymentService;
    private PluginClassLoaderManager pluginClassLoaderManager;

    private static File createTempDirectory()
            throws IOException
    {
        final File temp;

        temp = File.createTempFile("temp", Long.toString(System.nanoTime()));

        if (!(temp.delete()))
        {
            throw new IOException("Could not delete temp file: " + temp.getAbsolutePath());
        }

        if (!(temp.mkdir()))
        {
            throw new IOException("Could not create temp directory: " + temp.getAbsolutePath());
        }

        return (temp);
    }


    @Override
    public void dispose()
    {
        if (pluginManager != null)
        {
            pluginManager.dispose();
        }
    }

    @Override
    public void initialise() throws InitialisationException
    {
        pluginManager = createDefaultPluginManager();
        pluginManager.initialise();
        for ( ManagedMulePlugin plugin : pluginManager.registeredPlugins() ){
            if ( plugin.getPlugin() instanceof DeploymentListenerProvider) {
                deploymentService.addDeploymentListener(((DeploymentListenerProvider) plugin.getPlugin()).getDeploymentListener());
            }
        }
    }

    private PluginProcessor createDefaultPluginProcessor()
    {
        CompositePluginProcessor compositePluginProcessor = new CompositePluginProcessor();

        if (deploymentService != null)
        {
            compositePluginProcessor.addProcessor(new DeploymentListenerProviderProcessor(deploymentService));
            compositePluginProcessor.addProcessor(new DeploymentServiceAwareProcessor(deploymentService));
        }

        return compositePluginProcessor;
    }

    private PluginManager createDefaultPluginManager()
    {
        CompositePluginDiscoverer pluginDiscoverer = new CompositePluginDiscoverer();

        File pluginsFolder = getUserPluginFolder();
        if ( pluginsFolder != null ){
            pluginDiscoverer.addDiscoverer(new FileSystemPluginDiscoverer(pluginsFolder));
        }

        File debugerTempFolder = createDebuggerPluginTempFolder("mule.debug.enable", "mule-plugin-debugger.zip");
        if ( debugerTempFolder != null ){
            pluginDiscoverer.addDiscoverer(new FileSystemPluginDiscoverer(debugerTempFolder));
        }


        PluginFactory pluginFactory = createDefaultPluginFactory();
        MulePluginManagerFactory pluginManagerFactory = new MulePluginManagerFactory(pluginDiscoverer, pluginFactory);
        pluginManagerFactory.addPluginRegistrationListener(this);
        PluginProcessor pluginProcessor = createDefaultPluginProcessor();
        pluginManagerFactory.setPluginProcessor(pluginProcessor);

        return pluginManagerFactory.create();
    }

    @Override
    public String getName()
    {
        return "Mule Plugin Core Extension";
    }

    @Override
    public void start() throws MuleException
    {
        pluginManager.start();
    }

    @Override
    public void stop() throws MuleException
    {
        pluginManager.stop();
    }

    @Override
    public void setDeploymentService(DeploymentService deploymentService)
    {
        this.deploymentService = deploymentService;
    }

    public PluginManager getPluginManager()
    {
        return pluginManager;
    }

    @Override
    public void setPluginClassLoaderManager(PluginClassLoaderManager pluginClassLoaderManager)
    {
        this.pluginClassLoaderManager = pluginClassLoaderManager;
    }

    @Override
    public void onPluginRegistered(ManagedMulePlugin plugin)
    {
        ClassLoader pluginClassLoader = plugin.getPlugin().getClass().getClassLoader();
        PluginClassLoaderFilter filter = new PluginClassLoaderFilter(plugin.getDescriptor());
        FilteringPluginClassLoader filteringPluginClassLoader = new FilteringPluginClassLoader(plugin.getName(),pluginClassLoader, filter);

        pluginClassLoaderManager.addPluginClassLoader(filteringPluginClassLoader);
    }

    private File getUserPluginFolder()
    {
        File pluginsFolder = new File(getClass().getClassLoader().getResource("").getPath(), PLUGINS_DIR);
        if (!pluginsFolder.exists())
        {
            return null;
        }
        return pluginsFolder;
    }

    private PluginFactory createDefaultPluginFactory()
    {
        return new MulePluginFactory(new FileSystemPluginClassLoaderFactory(), new MulePluginPropertyDiscovererFactory(),
                                     new MulePluginConfigurerFactory());
    }

    private File createDebuggerPluginTempFolder(String enableCondition, String resourceName)
    {
        if (Boolean.getBoolean(enableCondition))
        {
            try
            {
                File tempDirectory = createTempDirectory();
                FileUtils.extractResources(resourceName, getClass(), tempDirectory, true);
                return tempDirectory;
            }
            catch (IOException e)
            {
                // Do nothing
            }
        }
        return null;
    }

}
