/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.plugins.coverage.extensions;

import com.mulesoft.mule.plugin.ManagedMulePlugin;
import com.mulesoft.mule.plugin.classloader.FileSystemPluginClassLoaderFactory;
import com.mulesoft.mule.plugin.classloader.FilteringPluginClassLoader;
import com.mulesoft.mule.plugin.classloader.PluginClassLoaderFilter;
import com.mulesoft.mule.plugin.discoverer.FileSystemPluginDiscoverer;
import com.mulesoft.mule.plugin.factory.MulePluginConfigurerFactory;
import com.mulesoft.mule.plugin.factory.MulePluginFactory;
import com.mulesoft.mule.plugin.factory.MulePluginPropertyDiscovererFactory;
import com.mulesoft.mule.plugin.factory.PluginFactory;
import com.mulesoft.mule.plugin.manager.MulePluginManagerFactory;
import com.mulesoft.mule.plugin.manager.PluginManager;
import com.mulesoft.mule.plugin.manager.PluginRegistrationListener;
import com.mulesoft.mule.plugin.processor.CompositePluginProcessor;
import com.mulesoft.mule.plugin.processor.PluginProcessor;
import com.mulesoft.mule.plugin.processor.deployment.DeploymentListenerProvider;
import com.mulesoft.mule.plugin.processor.deployment.DeploymentListenerProviderProcessor;
import com.mulesoft.mule.plugin.processor.deployment.DeploymentServiceAwareProcessor;
import org.apache.commons.lang.StringUtils;
import org.mule.MuleCoreExtension;
import org.mule.api.MuleException;
import org.mule.api.lifecycle.InitialisationException;
import org.mule.module.launcher.DeploymentService;
import org.mule.module.launcher.DeploymentServiceAware;
import org.mule.module.launcher.PluginClassLoaderManager;
import org.mule.module.launcher.PluginClassLoaderManagerAware;
import org.mule.util.FileUtils;

import java.io.File;
import java.io.IOException;

/**
 *
 */
public class MunitPluginCoreExtension implements MuleCoreExtension, DeploymentServiceAware, PluginClassLoaderManagerAware, PluginRegistrationListener {
//    private transient Log logger = LogFactory.getLog(this.getClass());

    public static final String PLUGINS_DIR = "mule-plugins";

    private static final String TEMP_DIRECTORY = "temp";
    private static final String ANYPOINT_STUDIO_DIRECTORY = "AnypointStudio";

    private PluginManager pluginManager;

    private DeploymentService deploymentService;
    private PluginClassLoaderManager pluginClassLoaderManager;

    @Override
    public void dispose() {
        if (pluginManager != null) {
            pluginManager.dispose();
        }
    }

    @Override
    public void initialise() throws InitialisationException {
        pluginManager = createDefaultPluginManager();
        pluginManager.initialise();
        for (ManagedMulePlugin plugin : pluginManager.registeredPlugins()) {
            if (plugin.getPlugin() instanceof DeploymentListenerProvider) {
                deploymentService.addDeploymentListener(((DeploymentListenerProvider) plugin.getPlugin()).getDeploymentListener());
            }
        }
    }

    private PluginProcessor createDefaultPluginProcessor() {
        CompositePluginProcessor compositePluginProcessor = new CompositePluginProcessor();

        if (deploymentService != null) {
            compositePluginProcessor.addProcessor(new DeploymentListenerProviderProcessor(deploymentService));
            compositePluginProcessor.addProcessor(new DeploymentServiceAwareProcessor(deploymentService));
        }

        return compositePluginProcessor;
    }

    private PluginManager createDefaultPluginManager() {
        CompositePluginDiscoverer pluginDiscoverer = new CompositePluginDiscoverer();

        File pluginsFolder = getUserPluginFolder();
        if (pluginsFolder != null) {
            pluginDiscoverer.addDiscoverer(new FileSystemPluginDiscoverer(pluginsFolder));
        }

//        logger.debug("Creating debugger temp folder...");
        File debugerTempFolder = createDebuggerPluginTempFolder("mule.debug.enable", "mule-plugin-debugger.zip");
        if (debugerTempFolder != null) {
            pluginDiscoverer.addDiscoverer(new FileSystemPluginDiscoverer(debugerTempFolder));
        } else {
//            logger.debug("Debugger temp folder creation failed");
        }


        PluginFactory pluginFactory = createDefaultPluginFactory();
        MulePluginManagerFactory pluginManagerFactory = new MulePluginManagerFactory(pluginDiscoverer, pluginFactory);
        pluginManagerFactory.addPluginRegistrationListener(this);
        PluginProcessor pluginProcessor = createDefaultPluginProcessor();
        pluginManagerFactory.setPluginProcessor(pluginProcessor);

        return pluginManagerFactory.create();
    }

    @Override
    public String getName() {
        return "Mule Plugin Core Extension";
    }

    @Override
    public void start() throws MuleException {
        pluginManager.start();
    }

    @Override
    public void stop() throws MuleException {
        pluginManager.stop();
    }

    @Override
    public void setDeploymentService(DeploymentService deploymentService) {
        this.deploymentService = deploymentService;
    }

    public PluginManager getPluginManager() {
        return pluginManager;
    }

    @Override
    public void setPluginClassLoaderManager(PluginClassLoaderManager pluginClassLoaderManager) {
        this.pluginClassLoaderManager = pluginClassLoaderManager;
    }

    @Override
    public void onPluginRegistered(ManagedMulePlugin plugin) {
        ClassLoader pluginClassLoader = plugin.getPlugin().getClass().getClassLoader();
        PluginClassLoaderFilter filter = new PluginClassLoaderFilter(plugin.getDescriptor());
        FilteringPluginClassLoader filteringPluginClassLoader = new FilteringPluginClassLoader(plugin.getName(), pluginClassLoader, filter);

        pluginClassLoaderManager.addPluginClassLoader(filteringPluginClassLoader);
    }

    private File getUserPluginFolder() {
//        logger.debug("Attempting to obtain plugins dir...");
        File pluginsFolder = new File(getClass().getClassLoader().getResource("").getPath(), PLUGINS_DIR);
        if (!pluginsFolder.exists()) {
//            logger.debug("Plugins dir not found");
            return null;
        }
        return pluginsFolder;
    }

    private PluginFactory createDefaultPluginFactory() {
        return new MulePluginFactory(new FileSystemPluginClassLoaderFactory(), new MulePluginPropertyDiscovererFactory(),
                new MulePluginConfigurerFactory());
    }

    private File createDebuggerPluginTempFolder(String enableCondition, String resourceName) {
        if (Boolean.getBoolean(enableCondition)) {
            try {
                File tempDirectory = createTempDirectory();
                FileUtils.extractResources(resourceName, getClass(), tempDirectory, true);
                return tempDirectory;
            } catch (IOException e) {
                // Do nothing
                e.printStackTrace();
            }
        }
        return null;
    }

    private File createTempDirectory() throws IOException {
        final File temp;

        String os = System.getProperty("os.name").toLowerCase();
        if (StringUtils.isNotBlank(os) && os.indexOf("win") >= 0) {
            temp = createWinTempDirectory();
        } else {
            temp = createLnxTempDirectory();
        }

        if (temp.exists()) {
            if (!(temp.delete())) {
                throw new IOException("Could not delete temp file: " + temp.getAbsolutePath());
            }
        }

        if (StringUtils.isNotBlank(os) && os.indexOf("win") >= 0) {
            if (!(temp.mkdirs())) {
                throw new IOException("Could not create temp directory: " + temp.getAbsolutePath());
            }
        } else {
            if (!(temp.mkdir())) {
                throw new IOException("Could not create temp directory: " + temp.getAbsolutePath());
            }
        }
        return (temp);
    }

    private void deleteWinTempDirectory() throws IOException {
        StringBuilder tempDirectoryPath = new StringBuilder();

        tempDirectoryPath
                .append(System.getProperty("user.home"))
                .append(File.separator)
                .append(ANYPOINT_STUDIO_DIRECTORY)
                .append(File.separator)
                .append(TEMP_DIRECTORY);


        File tempDirectory = new File(tempDirectoryPath.toString());
        if (tempDirectory.exists()) {
            org.apache.commons.io.FileUtils.deleteDirectory(tempDirectory);
        }
    }

    private File createWinTempDirectory() throws IOException {
        deleteWinTempDirectory();

        StringBuilder tempFolderPath = new StringBuilder();
        tempFolderPath
                .append(System.getProperty("user.home"))
                .append(File.separator)
                .append(ANYPOINT_STUDIO_DIRECTORY)
                .append(File.separator)
                .append(TEMP_DIRECTORY)
                .append(File.separator)
                .append(Long.toString(System.nanoTime()));

//        logger.debug("Creating temp folder at: " + tempFolderPath.toString());
        return new File(tempFolderPath.toString());
    }


    private File createLnxTempDirectory() throws IOException {
        StringBuilder tempFolderPath = new StringBuilder();
        tempFolderPath
                .append(TEMP_DIRECTORY)
                .append(Long.toString(System.nanoTime()));

        return File.createTempFile(tempFolderPath.toString(), "");
    }

}