/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.coverage;

import org.apache.maven.plugin.logging.Log;
import org.mule.api.MuleContext;
import org.mule.munit.plugins.coverage.CoberturaPlugin;
import org.mule.munit.plugins.coverage.PathBuilder;
import org.mule.munit.runner.MuleContextManager;
import org.mule.munit.runner.spring.config.model.MockingConfiguration;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;

/**
 * The goal of this class is to build the paths of a Mule application.
 * It require the list of Mule configuration resources to do so.
 */
public class ApplicationPathBuilder {
    private Log log;

    private String resources;
    private Set<String> flowPaths;
    private Set<String> subFlowPaths;
    private Set<String> batchPaths;

    public ApplicationPathBuilder(String resources) {
        this.resources = resources;
    }

    public String getResources() {
        return resources;
    }

    public Set<String> getFlowPaths() {
        if (null == flowPaths) {
            buildApplicationPaths();
        }

        return flowPaths;
    }

    public Set<String> getSubFlowPaths() {
        if (null == subFlowPaths) {
            buildApplicationPaths();
        }
        return subFlowPaths;
    }

    public Set<String> getBatchPaths() {
        if (null == batchPaths) {
            buildApplicationPaths();
        }
        return batchPaths;
    }


    /**
     * The method starts a Mule context ( by MUnit) and ones Mule has started it build the paths of all the flows
     * and sub-flows present in the application.
     */
    private void buildApplicationPaths() {
        debug("Counting application flow paths: " + new Date().toString());
        String originalCoveragePort = System.clearProperty(CoberturaPlugin.COBERTURA_PORT_PROPERTY);

        MuleContextManager muleContextManager = null;
        MuleContext muleContext = null;
        try {
            if (muleContext == null || muleContext.isDisposed()) {

                muleContextManager = new MuleContextManager(createConfiguration());
                muleContext = muleContextManager.createMule(resources);
                muleContext = muleContextManager.startMule(muleContext);

                PathBuilder pathBuilder = new PathBuilder(muleContext);
                flowPaths = new HashSet<String>(pathBuilder.buildFlowPathsMap().values());
                subFlowPaths = new HashSet<String>(pathBuilder.buildSubFlowPathsMap().values());
                batchPaths = new HashSet<String>(pathBuilder.buildBatchPathsMap().values());
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            muleContextManager.killMule(muleContext);

            if (null != originalCoveragePort) {
                System.setProperty(CoberturaPlugin.COBERTURA_PORT_PROPERTY, originalCoveragePort);
            }
            debug("Application flow paths count done: " + new Date().toString());
        }
    }

    private MockingConfiguration createConfiguration() {
        return new MockingConfiguration(true, new ArrayList<String>(), true, null);
    }

    public void setLog(Log log) {
        this.log = log;
    }

    private void info(String msg) {
        if (log != null) {
            log.info(msg);
        }
    }

    private void warn(String msg) {
        if (log != null) {
            log.warn(msg);
        }
    }

    private void debug(String msg) {
        if (log != null) {
            log.debug(msg);
        }
    }

    private void error(String msg) {
        if (log != null) {
            log.error(msg);
        }
    }

}
