/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.coverage;

import org.mule.coverage.report.MavenCoverageReport;
import org.mule.coverage.report.MuleFlow;
import org.mule.coverage.report.MuleResource;
import org.mule.log.MunitMavenLog;
import org.mule.munit.plugins.coverage.CoverageCalculator;
import org.mule.munit.plugins.coverage.PathBuilder;
import org.mule.util.MulePropertiesLoader;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.*;

public class CoverageReportBuilder {
    private MunitMavenLog log;
    private MulePropertiesLoader propertiesLoader;

    private Set<String> flowsToIgnore;

    public CoverageReportBuilder(MunitMavenLog log, MulePropertiesLoader propertiesLoader) {
        this.log = log;
        this.propertiesLoader = propertiesLoader;
    }

    public void setFlowsToIgnore(Set<String> flowsToIgnore) {
        this.flowsToIgnore = flowsToIgnore;
    }

    public MavenCoverageReport buildReport(Set<String> coveredPaths, Set<String> appFlowPaths, Set<String> appSubFlowPaths, Set<String> appBatchPaths) {

        coveredPaths = filterIgnoredFlows(coveredPaths);
        appFlowPaths = filterIgnoredFlows(appFlowPaths);
        appSubFlowPaths = filterIgnoredFlows(appSubFlowPaths);
        appBatchPaths = filterIgnoredFlows(appBatchPaths);


        Map<String, List<String>> filesMap = buildFilesFlowMap(propertiesLoader.getConfigResources());
        Map<String, List<String>> flowsMap = buildApplicationFlowPathMap(appFlowPaths, appSubFlowPaths, appBatchPaths);

        CoverageCalculator calculator = new CoverageCalculator(coveredPaths, appFlowPaths, appSubFlowPaths, appBatchPaths);
        CoverageCalculator.CoverageResult coverageResult = calculator.calculate();

        Map<String, List<String>> flowsCoveredPaths = PathBuilder.buildFlowPathsMap(coveredPaths);


        MavenCoverageReport report = new MavenCoverageReport();
        report.setCoverage(coverageResult.getCoverage());

        for (String fileName : filesMap.keySet()) {
            MuleResource muleResource = new MuleResource(fileName);

            List<String> flowsFromFile = filesMap.get(fileName);
            for (String flowName : flowsFromFile) {
                // Only Report flows that hasn't been ignored
                if (!flowsToIgnore.contains(flowName)) {
                    MuleFlow muleFlow = new MuleFlow(flowName);

                    if (flowsMap.containsKey(flowName)) {
                        muleFlow.getPaths().addAll(flowsMap.get(flowName));
                    }

                    if (flowsCoveredPaths.containsKey(flowName)) {
                        muleFlow.getCoveredPaths().addAll(flowsCoveredPaths.get(flowName));
                    }

                    muleResource.getFlows().add(muleFlow);
                }
            }

            report.getResources().add(muleResource);
        }


        return report;
    }

    private Set<String> filterIgnoredFlows(Set<String> paths) {
        if (null != flowsToIgnore) {
            return PathBuilder.filterPaths(paths, flowsToIgnore);
        }
        return paths;
    }

    private Map<String, List<String>> buildFilesFlowMap(String resources) {
        Map<String, List<String>> filesMap = new HashMap<String, List<String>>();
        log.debug("Building Files flow map...");

        if (resources != null && !resources.equals("")) {
            for (String resource : resources.split(",")) {
                String filePath = this.getClass().getClassLoader().getResource(resource).getPath();

                filesMap.put(resource, getFlowsFromFile(filePath));
            }
        }

        log.debug("Files flow map building done...");
        return filesMap;
    }

    private List<String> getFlowsFromFile(String filePath) {
        log.debug("Getting flows from file [" + filePath + "]");

        List<String> flowNames = new ArrayList<String>();

        String xpathQuery = "//*[local-name()='flow' or local-name()='sub-flow' or local-name()='job']";
        try {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            factory.setNamespaceAware(true);
            DocumentBuilder builder = factory.newDocumentBuilder();
            Document doc = builder.parse(new FileInputStream(filePath));

            XPath xpath = XPathFactory.newInstance().newXPath();

            NodeList nl = (NodeList) xpath.compile(xpathQuery).evaluate(doc, XPathConstants.NODESET);
            for (int i = 0; i < nl.getLength(); i++) {
                Node nameAttribute = nl.item(i).getAttributes().getNamedItem("name");
                flowNames.add(nameAttribute.getNodeValue());
            }

        } catch (ParserConfigurationException e) {
            // DO NOTHING
        } catch (SAXException e) {
            // DO NOTHING
        } catch (FileNotFoundException e) {
            // DO NOTHING
        } catch (XPathExpressionException e) {
            // DO NOTHING
        } catch (IOException e) {
            // DO NOTHING
        } finally {
            log.debug("Flow names loaded: " + flowNames);
            return flowNames;
        }
    }

    private Map<String, List<String>> buildApplicationFlowPathMap(Set<String> flows, Set<String> subFlows, Set<String> batches) {
        Map<String, List<String>> flowsMap = new HashMap<String, List<String>>();

        flowsMap.putAll(PathBuilder.buildFlowPathsMap(flows));
        flowsMap.putAll(PathBuilder.buildFlowPathsMap(subFlows));
        flowsMap.putAll(PathBuilder.buildFlowPathsMap(batches));

        return flowsMap;
    }
}
