/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.properties;

import org.apache.commons.lang3.Validate;

import java.util.HashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import java.util.Set;

/**
 * {@link MUnitUserPropertiesManager} allows to set the system properties
 * defined by the user for the execution of the MUnit tests. It also allows
 * to save and restore the state of the system properties on call.
 */
public class MUnitUserPropertiesManager {
    private static Set<String> notAllowedPropertyKeys = defineNotAllowedPropertyKeys();

    private Properties initialSystemProperties;

    public MUnitUserPropertiesManager() {
        storeInitialSystemProperties();
    }

    private static Set<String> defineNotAllowedPropertyKeys() {
        Set<String> propertyKeysSet = new HashSet<String>();

        propertyKeysSet.add("java.library.path");
        propertyKeysSet.add("file.encoding");
        propertyKeysSet.add("jdk.map.althashing.threshold");

        return propertyKeysSet;
    }

    public static Boolean isPropertyKeyAllowed(String key) {
        return !notAllowedPropertyKeys.contains(key);
    }


    public void storeInitialSystemProperties() {
        initialSystemProperties = (Properties) System.getProperties().clone();
    }

    public void addUserPropertiesToSystem(Map<String, String> userProperties) {
        if (userProperties != null) {
            for (Entry<String, String> entry : userProperties.entrySet()) {
                addUserPropertyToSystem(entry.getKey(), entry.getValue());
            }
        }
    }

    public void addUserPropertyToSystem(String key, String value) {
        Validate.notBlank(key, "The property key must not be null nor empty");
        if (isPropertyKeyAllowed(key)) {
            System.setProperty(key, value);
        }
    }

    public boolean hasProperty(String key){
        return System.getProperty(key) != null;
    }

    public void addApplicationPropertyToSystem(String key, String value) {
        Validate.notBlank(key, "The property key must not be null nor empty");
        System.setProperty(key, value);
    }

    public void restoreInitialSystemProperties() {
        System.setProperties(initialSystemProperties);
    }

}
