/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.properties;

import org.apache.maven.plugin.logging.Log;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

public class ApplicationPropertyLoader {
    private static final String DEFAULT_APP_FOLDER = "/src/main/app/";
    private static final String DEFAULT_APP_PROPERTIES_RESOURCE = "mule-app.properties";

    private Log log;
    private MUnitUserPropertiesManager propertiesManager;

    public ApplicationPropertyLoader(MUnitUserPropertiesManager propertiesManager, Log log) {
        this.log = log;
        this.propertiesManager = propertiesManager;
    }

    public void loadAndSetApplicationProperties(File projectBasedDir) {
        String appFolderPath = projectBasedDir + DEFAULT_APP_FOLDER;
        File appPropsFile = new File(appFolderPath, DEFAULT_APP_PROPERTIES_RESOURCE);

        try {
            Map<String, String> appPropsMap = loadApplicationProperties(appPropsFile);
            setApplicationProperties(appPropsMap);
        } catch (IOException e) {
            log.warn(DEFAULT_APP_PROPERTIES_RESOURCE + " could not be loaded.");
        }

    }

    private Map<String, String> loadApplicationProperties(File appPropsFile) throws IOException {
        Map<String, String> appPropsMap = new HashMap<String, String>();

        if (appPropsFile.exists() && appPropsFile.canRead()) {
            Properties props = loadPropertiesFromFile(appPropsFile.toURI().toURL());
            for (Object key : props.keySet()) {
                appPropsMap.put(key.toString(), props.getProperty(key.toString()));
            }
        }

        return appPropsMap;
    }

    private void setApplicationProperties(Map<String, String> properties) {
        for (Map.Entry<String, String> entry : properties.entrySet()) {
            if(!propertiesManager.hasProperty(entry.getKey())) {
                propertiesManager.addUserPropertyToSystem(entry.getKey(), entry.getValue());
                log.debug("System property [" + entry.getKey() + "] set to: [" + entry.getValue() + "]");
            }
        }
    }

    private Properties loadPropertiesFromFile(URL url) throws IOException {
        Properties props;
        if (url == null) {
            throw new IOException("Invalid file URL!");
        }
        InputStream is = url.openStream();
        try {
            props = new Properties();
            props.load(is);
        } finally {
            is.close();
        }
        return props;
    }


}
