/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.maven.model.Resource;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;
import org.mule.coverage.CoverageManager;
import org.mule.munit.common.util.FreePortFinder;
import org.mule.munit.runner.mule.MunitSuiteRunner;
import org.mule.munit.runner.mule.result.MunitResult;
import org.mule.munit.runner.mule.result.SuiteResult;
import org.mule.munit.runner.mule.result.notification.DummyNotificationListener;
import org.mule.munit.runner.mule.result.notification.NotificationListener;
import org.mule.munit.runner.output.DefaultOutputHandler;
import org.mule.notifiers.NotificationListenerDecorator;
import org.mule.notifiers.StreamNotificationListener;
import org.mule.notifiers.xml.XmlNotificationListener;
import org.mule.munit.runner.properties.MUnitUserPropertiesManager;

import java.io.*;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.*;

/**
 * Runs tests
 *
 * @goal test
 * @requiresDependencyResolution test
 * @goal test
 * @phase test
 */

public class MUnitMojo extends AbstractMojo {
    private static final String DEFAULT_MUNIT_BASE_SRC_FOLDER = "src/test/munit";
    public static final String TARGET_SUREFIRE_REPORTS_MUNIT_TXT = "/target/surefire-reports/munit.";
    public static final String TARGET_SUREFIRE_REPORTS_TEST_MUNIT_XML = "/target/surefire-reports/TEST-munit.";

    public static final String MUNIT_REPORT_FOLDER_PATH = "./target/munit-reports/";

    private static final String SKIP_TESTS_PROPERTY = "skipTests";
    private static final String SKIP_MUNIT_TESTS_PROPERTY = "skipMunitTests";


    private static final int MIN_PORT_NUMBER = 40000;
    private static final int MAX_PORT_NUMBER = 50000;

    public static final String SINGLE_TEST_NAME_TOKEN = "#";
    /**
     * @parameter expression="${project}"
     * @required
     */
    protected MavenProject project;

    /**
     * @parameter expression="${munit.test}"
     */
    protected String munittest;

    protected String testToRunName;

    /**
     * @parameter expression="${log.to.file}" default-value="false"
     */
    protected boolean logToFile;

    /**
     * List of System properties to pass to the MUnit tests.
     *
     * @parameter expression="${system.property.variables}"
     */
    protected Map<String, String> systemPropertyVariables;

    /**
     * List of System properties to assing a dynamic port value before start.
     *
     * @parameter expression="${dynamic.ports}"
     */
    protected List<String> dynamicPorts;

    /**
     * The classpath elements of the project being tested.
     *
     * @parameter expression="${project.testClasspathElements}"
     * @required
     * @readonly
     */
    protected List<String> classpathElements;

    /**
     * Define the behaviour of coverage.
     *
     * @parameter expression="${munit.coverage}"
     */
    protected Coverage coverage;

    /**
     * Manager for setting and restoring the user properties defined in the configuration
     */
    private MUnitUserPropertiesManager propertiesManager = new MUnitUserPropertiesManager();

    public void execute() throws MojoExecutionException {
        if (!"true".equals(System.getProperty(SKIP_TESTS_PROPERTY))) {

            if (!"true".equals(System.getProperty(SKIP_MUNIT_TESTS_PROPERTY))) {
                propertiesManager.storeInitialSystemProperties();
                try {
                    doExecute();
                } finally {
                    propertiesManager.restoreInitialSystemProperties();
                }
            } else {
                getLog().info("Run of munit-maven-plugin skipped. Property [" + SKIP_MUNIT_TESTS_PROPERTY + "] was set to true");
            }
        } else {
            getLog().info("Run of munit-maven-plugin skipped. Property [" + SKIP_TESTS_PROPERTY + "] was set to true");
        }
    }

    private void doExecute() throws MojoExecutionException {
        if (logToFile) {
            System.setProperty(DefaultOutputHandler.OUTPUT_FOLDER_PROPERTY, project.getBasedir() + TARGET_SUREFIRE_REPORTS_MUNIT_TXT + "%s-output.txt");
        }

        addSystemPropertyVariables();

        setDynamicPortValues();

        stopLicenseCheck();


        CoverageManager coverageManager = buildCoverageManager();

        List testResources = project.getTestResources();
        for (Object o : testResources) {
            Resource testResource = (Resource) o;
            testResource.getTargetPath();
        }

        try {
            List<SuiteResult> results = new ArrayList<SuiteResult>();
            addUrlsToClassPath(makeClassPath());
            File munitSourceFolder = new File(project.getBasedir(), DEFAULT_MUNIT_BASE_SRC_FOLDER);
            if (munitSourceFolder == null || !munitSourceFolder.exists()) {
                getLog().warn("The project has no " + DEFAULT_MUNIT_BASE_SRC_FOLDER + " folder. Aborting MUnit test run.");
                return;
            }

            Collection<File> allFiles = getMunitTestSuiteFileList(munitSourceFolder);
            for (File file : allFiles) {
                String fileName = file.getPath().replace(munitSourceFolder.getPath() + File.separator, "");

                parseFilter();
                if (fileName.endsWith(".xml") && validateFilter(fileName)) {
                    coverageManager.startCoberturaServer();

                    results.add(buildRunnerFor(fileName).run());

                    coverageManager.accountSuiteResults();
                } else {
                    getLog().debug("MUnit Test Suite file " + fileName + " skipped. It doesn't match filter criteria: [" + munittest + "]");
                }
            }

            coverageManager.printReport();
            Boolean resultSuccess = show(results);

            if (!resultSuccess) {
                throw new MojoExecutionException("Build Fail", new MojoExecutionException("MUnit Tests Failed"));
            }

            if (coverageManager.failBuild()) {
                throw new MojoExecutionException("Build Fail", new MojoFailureException("Coverage limits were not reached"));
            }

        } catch (MalformedURLException e) {
            e.printStackTrace();
        } catch (InvocationTargetException e) {
            e.printStackTrace();
        } catch (NoSuchMethodException e) {
            e.printStackTrace();
        } catch (IllegalAccessException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();  //To change body of catch statement use File | Settings | File Templates.
        } finally {
        }
    }

    private CoverageManager buildCoverageManager() {
        CoverageManager coverageManager = new CoverageManager(coverage, propertiesManager, getLog());

        return coverageManager;
    }

    private void stopLicenseCheck() {
        getLog().debug("Avoid license check for Mule EE components...");
        propertiesManager.addUserPropertyToSystem("mule.testingMode", "true");
    }

    private Collection<File> getMunitTestSuiteFileList(File munitTestFolder) throws FileNotFoundException {
        Collection<File> munitTestSuiteFiles = new ArrayList<File>();

        Collection<File> allFiles = FileUtils.listFiles(munitTestFolder, null, true);
        for (File file : allFiles) {
            if (isValidMunitTestSuiteFile(file)) {
                munitTestSuiteFiles.add(file);
            }
        }
        return munitTestSuiteFiles;
    }

    private boolean isValidMunitTestSuiteFile(File file) throws FileNotFoundException {
        String MUNIT_TEST_SUITE_FILE_MARKER = "munit:config";
        Scanner scanner = new Scanner(file);

        while (scanner.hasNextLine()) {
            String line = scanner.nextLine();
            if (line.contains(MUNIT_TEST_SUITE_FILE_MARKER)) {
                scanner.close();
                return true;
            }
        }

        scanner.close();
        return false;
    }

    private Boolean show(List<SuiteResult> results) throws MojoExecutionException {
        boolean success = true;

        System.out.println();
        System.out.println("===============================================================================");
        System.out.println("MUnit Run Summary                                                              ");
        System.out.println("===============================================================================");

        int testCount = 0;
        int errorCount = 0;
        int failCount = 0;
        int skipCount = 0;

        for (SuiteResult run : results) {
            List<MunitResult> failingTests = run.getFailingTests();
            List<MunitResult> errorTests = run.getErrorTests();

            System.out.println(" >> " + FilenameUtils.getName(run.getTestName()) + " test result: Tests: " + run.getNumberOfTests() + ", Errors: " + errorTests.size() + ", Failures:" + failingTests.size() + ", Skipped: " + run.getNumberOfSkipped());
            showFailures(failingTests);
            showError(errorTests);

            testCount += run.getNumberOfTests();
            errorCount += errorTests.size();
            failCount += failingTests.size();
            skipCount += run.getNumberOfSkipped();


            if (!failingTests.isEmpty() || !errorTests.isEmpty()) {
                success = false;
            }
        }

        System.out.println("\t");
        System.out.println("===============================================================================");
        System.out.println(" > Tests:\t" + testCount);
        System.out.println(" > Errors:\t" + errorCount);
        System.out.println(" > Failures:\t" + failCount);
        System.out.println(" > Skipped:\t" + skipCount);
        System.out.println("===============================================================================");

        return success;
    }

    private void showFailures(List<MunitResult> failingTests) {
        showUnsuccessfulTests(failingTests, "FAILED");
    }

    private void showError(List<MunitResult> errorTests) {
        showUnsuccessfulTests(errorTests, "ERROR");
    }

    private void showUnsuccessfulTests(List<MunitResult> unsuccessfulTests, String unsuccessfulTag) {
        if (!unsuccessfulTests.isEmpty()) {
            for (MunitResult result : unsuccessfulTests) {
                System.out.println("\t --- " + result.getTestName() + " <<< " + unsuccessfulTag);
            }
        }
    }

    private MunitSuiteRunner buildRunnerFor(String fileName) {
        List<String> testNameList = new ArrayList();
        if (StringUtils.isNotBlank(testToRunName)) {
            testNameList.add(testToRunName);
        }
        getLog().debug("Running MUnit Test Suite: " + fileName + ", for test names: " + testNameList);

        MunitSuiteRunner runner = new MunitSuiteRunner(fileName, testNameList);
        NotificationListenerDecorator listener = new NotificationListenerDecorator();
        listener.addNotificationListener(new StreamNotificationListener(System.out));
        listener.addNotificationListener(buildFileNotificationListener(fileName));
        listener.addNotificationListener(buildXmlNotificationListener(fileName));
        runner.setNotificationListener(listener);

        return runner;
    }

    private void addSystemPropertyVariables() {
        getLog().info("Setting system property variables...");
        if (null != systemPropertyVariables) {
            for (Map.Entry<String, String> entry : systemPropertyVariables.entrySet()) {
                if(!propertiesManager.hasSystemProperty(entry.getKey())) {
                    propertiesManager.addUserPropertyToSystem(entry.getKey(), entry.getValue());
                    getLog().debug("System property [" + entry.getKey() + "] set to: [" + entry.getValue() + "]");
                }
            }
        }
        getLog().debug("System property variables set [DONE]");
    }

    private void setDynamicPortValues() {
        getLog().info("Acquiring dynamic ports...");
        FreePortFinder portFinder = new FreePortFinder(MIN_PORT_NUMBER, MAX_PORT_NUMBER);
        if (null != dynamicPorts) {
            for (String portPlaceHolder : dynamicPorts) {
                Integer dynamicPort = portFinder.find();
                propertiesManager.addUserPropertyToSystem(portPlaceHolder, dynamicPort.toString());
                getLog().debug("Dynamic port [" + portPlaceHolder + "] set to: [" + dynamicPort.toString() + "]");
            }
        }
        getLog().debug("Dynamic port definition [DONE]");
    }

    private NotificationListener buildFileNotificationListener(String fileName) {
        fileName = fileName.replace(".xml", ".txt");
        fileName = fileName.replace('/', '.');
        try {
            return new StreamNotificationListener(new PrintStream(new FileOutputStream(getFile(project.getBasedir() + TARGET_SUREFIRE_REPORTS_MUNIT_TXT + fileName))));
        } catch (FileNotFoundException e) {
            e.printStackTrace();
            return new DummyNotificationListener();
        } catch (IOException e) {
            e.printStackTrace();
            return new DummyNotificationListener();
        }
    }

    private NotificationListener buildXmlNotificationListener(String fileName) {
        fileName = fileName.replace('/', '.');
        try {
            return new XmlNotificationListener(fileName, new PrintStream(new FileOutputStream(getFile(project.getBasedir() + TARGET_SUREFIRE_REPORTS_TEST_MUNIT_XML + fileName))));
        } catch (FileNotFoundException e) {
            e.printStackTrace();
            return new DummyNotificationListener();
        } catch (IOException e) {
            e.printStackTrace();
            return new DummyNotificationListener();
        }
    }

    private void parseFilter() {
        if (StringUtils.isNotBlank(munittest) && munittest.contains(SINGLE_TEST_NAME_TOKEN)) {
            testToRunName = munittest.substring(munittest.indexOf(SINGLE_TEST_NAME_TOKEN) + 1);
            munittest = munittest.substring(0, munittest.indexOf(SINGLE_TEST_NAME_TOKEN));
        } else {
            testToRunName = "";
        }

    }

    private boolean validateFilter(String fileName) {
        if (munittest == null) {
            return true;
        }

        return fileName.matches(munittest);
    }

    public URLClassLoader getClassPath(List<URL> classpath) {
        return new URLClassLoader(classpath.toArray(new URL[classpath.size()]), getClass().getClassLoader());
    }

    /**
     * Creates a classloader for loading tests.
     * <p/>
     * <p/>
     * We need to be able to see the same JUnit classes between this code and the mtest code,
     * but everything else should be isolated.
     */
    private List<URL> makeClassPath() throws MalformedURLException {

        List<URL> urls = new ArrayList<URL>(classpathElements.size());

        for (String e : classpathElements) {
            urls.add(new File(e).toURL());
        }
        return urls;
    }

    private void addUrlsToClassPath(List<URL> urls) throws NoSuchMethodException, InvocationTargetException, IllegalAccessException {
        ClassLoader sysCl = Thread.currentThread().getContextClassLoader();
        Class refClass = URLClassLoader.class;
        Method methodAddUrl = refClass.getDeclaredMethod("addURL", new Class[]{URL.class});
        methodAddUrl.setAccessible(true);
        for (Iterator it = urls.iterator(); it.hasNext(); ) {
            URL url = (URL) it.next();
            methodAddUrl.invoke(sysCl, url);
        }
    }

    private File getFile(String fullPath) throws IOException {
        File file = new File(fullPath);

        if (!file.getParentFile().exists()) {
            if (!file.getParentFile().mkdir()) {
                throw new IOException("Failed to create directory " + file.getParent());
            }
        }

        if (!file.exists()) {
            if (!file.createNewFile()) {
                throw new IOException("Failed to create file " + file.getName());
            }
        }

        return file;
    }
}
