/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.coverage.printer;

import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;
import org.mule.Coverage;
import org.mule.MUnitMojo;
import org.mule.coverage.CoverageManager;
import org.mule.coverage.report.MavenCoverageReport;
import org.mule.coverage.report.MuleResource;

import java.io.*;


public class HtmlPrinter implements MavenCoverageReportPrinter {
    public static final String PRINTER_NAME = "HTML";


    private static final String REPORT_FOLDER_PATH = MUnitMojo.MUNIT_REPORT_FOLDER_PATH + CoverageManager.COVERAGE_NAME_FOLDER + "/";
    private static final String ASSETS_FOLDER = "assets/";
    private static final String STYLES_FOLDER = "styles/";
    private static final String SCRIPTS_FOLDER = "js/";
    private static final String ICONS_FOLDER = "icons/";
    private static final String FONTS_FOLDER = "fonts/";

    private static final String SUMMARY_TEMPLATE_FILE_NAME = "summary-template.html";
    private static final String RESOURCE_TEMPLATE_FILE_NAME = "resource-template.html";

    private static final String MULESOFT_STYLES = "mulesoft-styles.css";
    private static final String TSORTER_STYLES = "tsorter.css";
    private static final String MULESOFT_ICON = ICONS_FOLDER + "004_logo.svg";
    private static final String MULESOFT_ICON_TTF = FONTS_FOLDER + "muleicons.ttf";

    private static final String TSORTER_SCRIPT = "tsorter.min.js";

    private Coverage configuration;

    private ReportFileWriter reportFileWriter = new ReportFileWriter();

    public HtmlPrinter(Coverage configuration){
        this.configuration = configuration;
    }

    @Override
    public String getPrinterName() {
        return PRINTER_NAME;
    }

    @Override
    public void printReport(MavenCoverageReport report) {
        try {
            printSummary(report);
            printResources(report);
            addAssets();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public void setReportFileWriter(ReportFileWriter reportFileWriter) {
        this.reportFileWriter = reportFileWriter;
    }

    private void printSummary(MavenCoverageReport report) throws IOException {
        Velocity.init();

        Reader reader = null;
        InputStream inputStream = null;
        StringWriter writer = null;
        try {
            inputStream = this.getClass().getClassLoader().getResourceAsStream(SUMMARY_TEMPLATE_FILE_NAME);
            reader = new InputStreamReader(inputStream, "UTF-8");

            VelocityContext context = new VelocityContext();
            context.put("applicationFlowCount", report.getApplicationFlowCount());
            context.put("applicationCoverage", String.format("%.2f",report.getCoverage()));
            context.put("applicationResourceLines", report.getResources());
            context.put("configuration",configuration);

            writer = new StringWriter();
            Velocity.evaluate(context, writer, "Rendering Coverage Summary Report", reader);

            reportFileWriter.writeToFile(REPORT_FOLDER_PATH + "summary.html", writer.getBuffer().toString());
        } catch (UnsupportedEncodingException e) {
            e.printStackTrace();
        } finally {
            if (null != reader) {
                reader.close();
            }
            if (null != inputStream) {
                inputStream.close();
            }
            if (null != writer) {
                writer.close();
            }
        }
    }

    private void printResources(MavenCoverageReport report) throws IOException {
        Velocity.init();

        Reader reader = null;
        StringWriter writer = null;
        InputStream inputStream = null;

        for (MuleResource mr : report.getResources()) {
            try {
                inputStream = this.getClass().getClassLoader().getResourceAsStream(RESOURCE_TEMPLATE_FILE_NAME);
                reader = new InputStreamReader(inputStream, "UTF-8");

                VelocityContext context = new VelocityContext();

                context.put("resourceName", mr.getName());
                context.put("resourceMpCovered", Math.round(mr.getCoverage()*mr.getMessageProcessorCount()/100));
                context.put("resourceMpCount", mr.getMessageProcessorCount());
                context.put("resourceCoverage", String.format("%.2f", mr.getCoverage()));
                context.put("flowLines", mr.getFlows());
                context.put("configuration",configuration);

                writer = new StringWriter();
                Velocity.evaluate(context, writer, "Rendering Flow Report", reader);

                reportFileWriter.writeToFile(REPORT_FOLDER_PATH + reportFileWriter.getHtmlFileName(mr.getName()), writer.getBuffer().toString());
            } catch (UnsupportedEncodingException e) {
                e.printStackTrace();
            } finally {
                if (null != reader) {
                    reader.close();
                }
                if (null != inputStream) {
                    inputStream.close();
                }

                if (null != writer) {
                    writer.close();

                }
            }
        }
    }

    private void addAssets(){
        addStyles();
        addScripts();
    }

    private void addStyles(){
        try {
            String stylesPath = REPORT_FOLDER_PATH + ASSETS_FOLDER + STYLES_FOLDER;
            reportFileWriter.copyFile(MULESOFT_STYLES, stylesPath + MULESOFT_STYLES);
            reportFileWriter.copyFile(MULESOFT_ICON, stylesPath + MULESOFT_ICON);
            reportFileWriter.copyFile(MULESOFT_ICON_TTF, stylesPath + MULESOFT_ICON_TTF);
            reportFileWriter.copyFile(TSORTER_STYLES, stylesPath + TSORTER_STYLES);
        } catch (IOException e){
            e.printStackTrace();
        }
    }

    private void addScripts(){
        try {
            String scriptsPath = REPORT_FOLDER_PATH + ASSETS_FOLDER + SCRIPTS_FOLDER;
            reportFileWriter.copyFile(TSORTER_SCRIPT, scriptsPath + TSORTER_SCRIPT);
        } catch (IOException e){
            e.printStackTrace();
        }
    }
}
