/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.coverage.server;

import com.google.gson.Gson;
import org.apache.maven.plugin.logging.Log;
import org.mule.log.MunitMavenLog;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketTimeoutException;

public class MunitCoverageServer implements Runnable {

    // 30 mins in millis
    private static final int SOCKET_TIMEOUT_MILLIS = 1800000;

    private MunitMavenLog log;

    private int port;

    private boolean running;

    private CoveragePluginReport report;

    public MunitCoverageServer(int port) {
        this.port = port;
    }

    public synchronized void launch() {
        new Thread(this).start();
    }

    public void run() {
        ObjectInputStream in = null;
        ServerSocket providerSocket = null;

        try {
            providerSocket = new ServerSocket(port, 10);
            log.debug("Coverage server awaiting for coverage connection in port [" + port + "]...");

            // TODO: WE MAY NEED TO CHANGE THIS ALL TOGETHER
            // This time out is large as the coverage plugins sends its report at the end of the run which could take some time
            providerSocket.setSoTimeout(SOCKET_TIMEOUT_MILLIS);

            Socket connection = providerSocket.accept();

            log.debug("Coverage server connection received from " + connection.getInetAddress().getHostName());

            in = new ObjectInputStream(connection.getInputStream());

            do {
                running = true;
                try {
                    String message = (String) in.readObject();
                    Gson gson = new Gson();
                    report = gson.fromJson(message, CoveragePluginReport.class);

                    log.debug("Coverage report received");
                } catch (ClassNotFoundException classNot) {
                    log.error("Report deserialization fail, no report will be generated");
                }

            } while (true);
        } catch (SocketTimeoutException timeoutException) {
            log.error("Coverage server time out, no report received. No report will be generated");
        } catch (IOException ioException) {
            // DO NOTHING
        } finally {
            try {
                log.debug("Shutting down coverage server running in port [" + port + "]...");

                running = false;

                if (null != in) {
                    in.close();
                }

                if (null != providerSocket) {
                    providerSocket.close();
                }

                log.debug("Coverage server shutdown");
            } catch (IOException ioException) {
                log.debug("Coverage server error during shutdown.");
            }
        }
    }


    public int getPort() {
        return this.port;
    }

    public boolean isRunning() {
        return running;
    }

    public CoveragePluginReport getReport() {
        return report;
    }

    public void setLog(Log log) {
        this.log = new MunitMavenLog(log, "CoverageServer");
    }

}
