/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.coverage;

import org.apache.maven.plugin.logging.Log;
import org.mule.Coverage;
import org.mule.MUnitMojo;
import org.mule.MuleRuntimeDetector;
import org.mule.coverage.printer.*;
import org.mule.log.MunitMavenLog;
import org.mule.munit.common.util.FreePortFinder;
import org.mule.munit.plugins.coverage.CoberturaPlugin;
import org.mule.munit.plugins.coverage.report.ApplicationCoverageReport;
import org.mule.munit.plugins.coverage.report.CoverageReportBuilder;
import org.mule.munit.plugins.coverage.report.SingleRunReport;
import org.mule.munit.plugins.coverage.report.model.MuleFlow;
import org.mule.munit.plugins.coverage.report.model.MuleResource;
import org.mule.munit.plugins.coverage.server.MunitCoverageServer;
import org.mule.munit.plugins.coverage.server.ReportAccumulator;
import org.mule.munit.runner.properties.MUnitUserPropertiesManager;
import org.mule.util.MulePropertiesLoader;

import java.io.File;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

public class CoverageManager implements ReportAccumulator {
    private static final int MIN_PORT_NUMBER = 50000;
    private static final int MAX_PORT_NUMBER = 55000;

    public static final String COVERAGE_NAME_FOLDER = "coverage";
    public static final String JSON_COVERAGE_NAME_FOLDER = "coverage-json";


    private MunitMavenLog log;
    private MuleRuntimeDetector muleRuntimeDetector = new MuleRuntimeDetector();

    private Coverage coverageConfiguration;

    private Integer coveragePort;
    private MunitCoverageServer coverageServer;
    private String projectName ;

    private Set<String> coveredPaths = new HashSet<String>();

    private List<MavenCoverageReportPrinter> printers = new ArrayList<MavenCoverageReportPrinter>();

    private ApplicationCoverageReport coverageReport = null;

    public CoverageManager(Coverage coverageConfiguration,  String projectName, Log log) {
        this.coverageConfiguration = coverageConfiguration;
        this.log = new MunitMavenLog(log, "CoverageManager");
        this.printers = new ArrayList<MavenCoverageReportPrinter>();

        this.projectName = projectName;

        this.setUpPrinters();
    }

    public void setMuleRuntimeDetector(MuleRuntimeDetector muleRuntimeDetector) {
        this.muleRuntimeDetector = muleRuntimeDetector;
    }

    private boolean isRunningOnEE() {
        return muleRuntimeDetector.isRunningOnEE();
    }

    private Boolean shouldRunCoverage() {
        if (!isRunningOnEE()) {
            return false;
        }

        if (coverageConfiguration == null) {
            return false;
        }

        return coverageConfiguration.shouldRunCoverage();
    }

    public void startCoverageServer() {
        if (!shouldRunCoverage()) {
            if (!isRunningOnEE()) {
                log.info("Coverage feature runs over EE runtime only. Feature is shut down.");
            } else {
                log.info("Coverage feature hasn't been enable. Not Gathering coverage data.");
            }
            return;
        }

        acquireCoverageServerPort();
        coverageServer = new MunitCoverageServer(coveragePort, this);

        log.info("Starting coverage server in port [" + coveragePort + "]");
        coverageServer.launch();
    }

    public void stopCoverageServer() {
        if (!shouldRunCoverage()) {
            return;
        }
        coverageServer.shutdown();
    }

    @Override
    public synchronized void accumulateReport(SingleRunReport report) {
        if (!shouldRunCoverage()) {
            return;
        }

        log.debug("Accounting suite coverage data...");
        if (report != null) {
            log.debug("Adding all covered paths");
            coveredPaths.addAll(report.getCoveredPaths());
        } else {
            log.debug("Accounting coverage data of suite fail. Not report found.");
        }
    }

    public Boolean failBuild() {
        if (!shouldRunCoverage()) {
            return false;
        }

        if (!coverageConfiguration.getFailBuild()) {
            return false;
        }


        if (coverageConfiguration.getRequiredApplicationCoverage() != -1) {
            if (getReport().getCoverage() < coverageConfiguration.getRequiredApplicationCoverage()) {
                return true;
            }
        }

        if (coverageConfiguration.getRequiredResourceCoverage() != -1) {
            for (MuleResource mr : getReport().getResources()) {
                if (mr.getCoverage() != -1 && mr.getCoverage() < coverageConfiguration.getRequiredResourceCoverage()) {
                    return true;
                }
            }
        }

        if (coverageConfiguration.getRequiredFlowCoverage() != -1) {
            for (MuleResource mr : getReport().getResources()) {
                for (MuleFlow mf : mr.getFlows()) {
                    if (mf.getCoverage() < coverageConfiguration.getRequiredFlowCoverage()) {
                        return true;
                    }
                }
            }
        }

        return false;
    }

    public void printReport() {
        if (!shouldRunCoverage()) {
            return;
        }

        ApplicationCoverageReport report = getReport();
        if (printers.size() > 0) {
            log.info("Printing Coverage Report...");
            createCoverageReportFolder();
        } else {
            log.info("No printers defined. The Coverage Report won't be printed.");
        }

        for (MavenCoverageReportPrinter printer : printers) {
            log.debug("Executing: " + printer.getPrinterName() + " printer...");
            printer.printReport(report);
        }

        SummaryConsolePrinter defaultPrinter = new SummaryConsolePrinter(log.getLog());
        defaultPrinter.setRequiredApplicationCoverage(coverageConfiguration.getRequiredApplicationCoverage());
        defaultPrinter.setRequiredResourceCoverage(coverageConfiguration.getRequiredResourceCoverage());
        defaultPrinter.setRequiredFlowCoverage(coverageConfiguration.getRequiredFlowCoverage());
        defaultPrinter.setFailBuild(coverageConfiguration.getFailBuild());
        defaultPrinter.printReport(report);
    }

    private void setUpPrinters() {
        for (String printFormat : coverageConfiguration.getFormats()) {
            if (ConsolePrinter.PRINTER_NAME.toLowerCase().equals(printFormat.toLowerCase())) {
                printers.add(new ConsolePrinter(log.getLog()));
                continue;
            }

            if (HtmlPrinter.PRINTER_NAME.toLowerCase().equals(printFormat.toLowerCase())) {
                printers.add(new HtmlPrinter(coverageConfiguration));
                continue;
            }

            if (JsonPrinter.PRINTER_NAME.toLowerCase().equals(printFormat.toLowerCase())) {
                printers.add(new JsonPrinter(coverageConfiguration));
                continue;
            }

            log.debug("No report format find for: " + printFormat);
        }
    }


    private void acquireCoverageServerPort() {
        FreePortFinder portFinder = new FreePortFinder(MIN_PORT_NUMBER, MAX_PORT_NUMBER);
        if (coveragePort == null || (!portFinder.isPortFree(coveragePort))) {
            coveragePort = portFinder.find();
        }
        setCoveragePortProperty();
    }

    private void setCoveragePortProperty() {
        MUnitUserPropertiesManager.addUserPropertyToSystem(CoberturaPlugin.COBERTURA_PORT_PROPERTY, coveragePort.toString());
    }

    protected ApplicationCoverageReport getReport() {
        if (null == coverageReport) {
            MulePropertiesLoader propertiesLoader = new MulePropertiesLoader();

            ApplicationPathBuilder appPathBuilder = new ApplicationPathBuilder(propertiesLoader.getConfigResources(), projectName);
            appPathBuilder.setLog(log);

            CoverageReportBuilder reportBuilder = new CoverageReportBuilder(propertiesLoader.getConfigResources());
            Set<String> flowsToIgnore = new HashSet<String>(coverageConfiguration.getIgnoreFlows());
            reportBuilder.setFlowsToIgnore(flowsToIgnore);

            this.coverageReport = reportBuilder.buildReport(coveredPaths, appPathBuilder.getFlowPaths(), appPathBuilder.getSubFlowPaths(), appPathBuilder.getBatchPaths());
        }
        return coverageReport;
    }

    private void createCoverageReportFolder() {
        log.debug("Creating coverage folder...");
        new File(MUnitMojo.MUNIT_REPORT_FOLDER_PATH + COVERAGE_NAME_FOLDER).mkdirs();
    }
}
