/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.util;

import org.apache.maven.plugin.logging.Log;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Enumeration;
import java.util.Properties;

/**
 * This class knows how to access several properties file that belongs (by default) to a Mule application.
 * It also provide specific methods to retrieve particular propreties.
 */
public class MulePropertiesLoader {
    private static final String MULE_DEPLOY_PROPERTIES = "mule-deploy.properties";
    private static final String MULE_DEPLOY_PROPERTIES_RELATIVE_PATH = File.separator + "target" + File.separator + "classes" + File.separator + MULE_DEPLOY_PROPERTIES;
    private static final String MULE_DEPLOY_PROPERTIES_TEST_RELATIVE_PATH = File.separator + "target" + File.separator + "test-classes" + File.separator + MULE_DEPLOY_PROPERTIES;

    private transient Log log;
    private String projectName;

    public MulePropertiesLoader(String projectName, Log log) {
        this.log = log;
        this.projectName = projectName;
    }

    /**
     * It'll return the value of the property config.resources in the file mule-deploy.properties
     *
     * @return the config resources or an empty string.
     */
    public String getConfigResources() {
        Properties props = this.loadProperties();

        if (props != null && props.getProperty("config.resources") != null) {
            return props.getProperty("config.resources");
        } else {
            return "";
        }
    }

    private Properties loadProperties() {
        try {
            Properties prop = new Properties();

            URL muleDeployPropertiesUrl = findMuleDeployProperties();
            if (null == muleDeployPropertiesUrl) {
                return null;
            }

            InputStream in = muleDeployPropertiesUrl.openStream();
            prop.load(in);
            in.close();
            return prop;
        } catch (Throwable t) {
            return null;
        }
    }

    private URL findMuleDeployProperties() throws IOException {
        URL muleDeployPropertiesUrl = null;
        Enumeration<URL> resources = getClass().getClassLoader().getResources(MULE_DEPLOY_PROPERTIES);

        while (resources.hasMoreElements()) {

            URL url = resources.nextElement();
            if (doesTheFileBelongToTheApplication(url)) {
                log.debug("Found " + MULE_DEPLOY_PROPERTIES + " file at: " + url.getPath());
                muleDeployPropertiesUrl = url;
                break;
            }

        }

        if (null == muleDeployPropertiesUrl) {
            log.debug("Failed to find " + MULE_DEPLOY_PROPERTIES + " file");
        }

        return muleDeployPropertiesUrl;
    }

    private boolean doesTheFileBelongToTheApplication(URL url) {
        String path = new File(url.getPath()).getPath();
        if (path.contains(projectName + MULE_DEPLOY_PROPERTIES_RELATIVE_PATH) ||
                path.contains(projectName + MULE_DEPLOY_PROPERTIES_TEST_RELATIVE_PATH)) {
            return true;
        }
        return false;
    }
}
