/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.coverage;

import org.apache.maven.plugin.logging.Log;
import org.mule.Coverage;
import org.mule.MuleRuntimeDetector;
import org.mule.coverage.printer.*;
import org.mule.munit.plugins.coverage.report.ApplicationCoverageReport;
import org.mule.munit.plugins.coverage.report.model.MuleFlow;
import org.mule.munit.plugins.coverage.report.model.MuleResource;

import java.io.File;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

public class CoverageManager {

    public static final String COVERAGE_NAME_FOLDER = "coverage";
    public static final String JSON_COVERAGE_NAME_FOLDER = "coverage-json";


    private Log log;
    private MuleRuntimeDetector muleRuntimeDetector = new MuleRuntimeDetector();

    private Coverage coverageConfiguration;

    private List<MavenCoverageReportPrinter> printers = new ArrayList<MavenCoverageReportPrinter>();

    private ApplicationCoverageReport coverageReport = null;

    private File coverageReportsFolder;

    public CoverageManager(Coverage coverageConfiguration, File coverageReportsFolder, Log log) {
        this.coverageConfiguration = coverageConfiguration;
        this.log = log;
        this.printers = new ArrayList<MavenCoverageReportPrinter>();

        this.coverageReportsFolder = coverageReportsFolder;

        this.setUpPrinters();
    }

    public void setMuleRuntimeDetector(MuleRuntimeDetector muleRuntimeDetector) {
        this.muleRuntimeDetector = muleRuntimeDetector;
    }

    private boolean isRunningOnEE() {
        return muleRuntimeDetector.isRunningOnEE();
    }

    public void setClasspath(List<String> classpath) {
        muleRuntimeDetector.setClasspath(classpath);
    }

    public Boolean shouldRunCoverage() {
        if (!isRunningOnEE()) {
            return false;
        }

        if (coverageConfiguration == null) {
            return false;
        }

        return coverageConfiguration.shouldRunCoverage();
    }

    public Set<String> getIgnoredFlows() {
        Set<String> ignoredFlows = new HashSet<String>();
        if (coverageConfiguration.getIgnoreFlows() != null) {
            ignoredFlows.addAll(new HashSet<String>(coverageConfiguration.getIgnoreFlows()));
        }
        return ignoredFlows;
    }

    public Boolean failBuild() {
        if (!shouldRunCoverage()) {
            return false;
        }

        if (!coverageConfiguration.getFailBuild()) {
            return false;
        }

        if (coverageConfiguration.getRequiredApplicationCoverage() != -1) {
            if (getReport().getCoverage() < coverageConfiguration.getRequiredApplicationCoverage()) {
                return true;
            }
        }

        if (coverageConfiguration.getRequiredResourceCoverage() != -1) {
            for (MuleResource mr : getReport().getResources()) {
                if (mr.getCoverage() != -1 && mr.getCoverage() < coverageConfiguration.getRequiredResourceCoverage()) {
                    return true;
                }
            }
        }

        if (coverageConfiguration.getRequiredFlowCoverage() != -1) {
            for (MuleResource mr : getReport().getResources()) {
                for (MuleFlow mf : mr.getFlows()) {
                    if (mf.getCoverage() < coverageConfiguration.getRequiredFlowCoverage()) {
                        return true;
                    }
                }
            }
        }

        return false;
    }

    public void printReport() {
        if (!shouldRunCoverage()) {
            return;
        }

        ApplicationCoverageReport report = getReport();
        if (printers.size() > 0) {
            log.info("Printing Coverage Report...");
            createCoverageReportFolder();
        } else {
            log.info("No printers defined. The Coverage Report won't be printed.");
        }

        for (MavenCoverageReportPrinter printer : printers) {
            log.debug("Executing: " + printer.getPrinterName() + " printer...");
            printer.printReport(report);
        }

        SummaryConsolePrinter defaultPrinter = new SummaryConsolePrinter(log);
        defaultPrinter.setRequiredApplicationCoverage(coverageConfiguration.getRequiredApplicationCoverage());
        defaultPrinter.setRequiredResourceCoverage(coverageConfiguration.getRequiredResourceCoverage());
        defaultPrinter.setRequiredFlowCoverage(coverageConfiguration.getRequiredFlowCoverage());
        defaultPrinter.setFailBuild(coverageConfiguration.getFailBuild());
        defaultPrinter.printReport(report);
    }

    private void setUpPrinters() {
        if (coverageConfiguration == null || coverageConfiguration.getFormats() == null) {
            return;
        }
        for (String printFormat : coverageConfiguration.getFormats()) {
            if (ConsolePrinter.PRINTER_NAME.toLowerCase().equals(printFormat.toLowerCase())) {
                printers.add(new ConsolePrinter(log));
                continue;
            }

            if (HtmlPrinter.PRINTER_NAME.toLowerCase().equals(printFormat.toLowerCase())) {
                printers.add(new HtmlPrinter(coverageConfiguration, coverageReportsFolder.getAbsolutePath() + File.separator));
                continue;
            }

            if (JsonPrinter.PRINTER_NAME.toLowerCase().equals(printFormat.toLowerCase())) {
                printers.add(new JsonPrinter(coverageConfiguration, coverageReportsFolder.getAbsolutePath() + File.separator));
                continue;
            }

            log.debug("No report format find for: " + printFormat);
        }
    }

    protected ApplicationCoverageReport getReport() {
        return coverageReport;
    }

    public void setReport(ApplicationCoverageReport coverageReport) {
        this.coverageReport = coverageReport;
    }

    private void createCoverageReportFolder() {
        log.debug("Creating coverage folder...");
        new File(coverageReportsFolder, COVERAGE_NAME_FOLDER).mkdirs();
    }
}
