/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.*;
import org.apache.maven.project.MavenProject;
import org.apache.maven.shared.utils.cli.CommandLineException;
import org.apache.maven.shared.utils.cli.CommandLineUtils;
import org.apache.maven.toolchain.Toolchain;
import org.apache.maven.toolchain.ToolchainManager;
import org.mule.coverage.CoverageManager;
import org.mule.munit.remote.RemoteRunner;
import org.mule.runner.ErrorStreamConsumer;
import org.mule.runner.JVMStarter;
import org.mule.runner.RunnerStreamConsumer;
import org.mule.runner.StartupConfiguration;
import org.mule.runner.handler.RunnerMessageHandler;
import org.mule.runner.handler.StandardOutputMessageHandler;
import org.mule.runner.handler.TestOutputMessageHandler;
import org.mule.runner.model.RunResult;
import org.mule.runner.printer.*;
import org.mule.util.ClasspathManager;
import org.mule.util.MulePropertiesLoader;
import org.mule.util.SystemPropertiesManager;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.net.MalformedURLException;
import java.util.*;


@Mojo(name = "test", defaultPhase = LifecyclePhase.TEST, requiresDependencyResolution = ResolutionScope.TEST)
public class MUnitMojo extends AbstractMojo {

    public static final String SINGLE_TEST_NAME_TOKEN = "#";
    private static final Class REMOTE_RUNNER_CLASS = RemoteRunner.class;

    private static final String[] MUNIT_SUITE_EXTENSIONS = new String[]{"xml"};

    private static final String SKIP_TESTS_PROPERTY = "skipTests";
    private static final String SKIP_MUNIT_TESTS_PROPERTY = "skipMunitTests";
    private static final String SKIP_ZIP_LOADING_PROPERTY = "skipZipLoading";

    protected String testToRunName;

    @Parameter(property = "project", required = true)
    protected MavenProject project;

    @Parameter(property = "munit.test")
    protected String munittest;

    @Deprecated
    @Parameter(property = "log.to.file", defaultValue = "false")
    protected boolean logToFile;

    @Parameter(property = "system.property.variables")
    protected Map<String, String> systemPropertyVariables;

    @Parameter(property = "environment.variables")
    protected Map<String, String> environmentVariables;

    @Parameter(property = "dynamic.ports")
    protected List<String> dynamicPorts;

    @Parameter(property = "project.testClasspathElements", required = true, readonly = true)
    protected List<String> classpathElements;

    @Parameter(property = "munit.coverage")
    protected Coverage coverage;

    @Parameter(property = "argLines")
    protected List<String> argLines;

    @Parameter(property = "additionalClasspathElements")
    protected List<String> additionalClasspathElements;

    @Parameter(property = "classpathDependencyExcludes")
    protected List<String> classpathDependencyExcludes;

    @Parameter(defaultValue = "${project.basedir}/src/test/munit/")
    protected File munitTestsDirectory;

    @Parameter(defaultValue = "${project.build.directory}/munit-reports/")
    protected File munitReportsDirectory;

    @Parameter(property = "redirectTestOutputToFile", defaultValue = "false")
    protected boolean redirectTestOutputToFile;

    @Parameter(defaultValue = "${project.build.directory}/munit-reports/output/")
    protected File testOutputDirectory;

    @Parameter(property = "enableSurefireReports", defaultValue = "true")
    protected boolean enableSurefireReports;

    @Parameter(defaultValue = "${project.build.directory}/surefire-reports/")
    protected File surefireReportsFolder;

    @Parameter(defaultValue = "${project.build.directory}/.mule")
    protected File muleWorkingDirectory;

    @Component
    protected MavenSession session;

    @Component
    protected ToolchainManager toolchainManager;

    public void execute() throws MojoExecutionException {
        if (!"true".equals(System.getProperty(SKIP_TESTS_PROPERTY))) {

            if (!"true".equals(System.getProperty(SKIP_MUNIT_TESTS_PROPERTY))) {
                doExecute();
            } else {
                getLog().info("Run of munit-maven-plugin skipped. Property [" + SKIP_MUNIT_TESTS_PROPERTY + "] was set to true");
            }
        } else {
            getLog().info("Run of munit-maven-plugin skipped. Property [" + SKIP_TESTS_PROPERTY + "] was set to true");
        }
    }

    private void doExecute() throws MojoExecutionException {

        if (logToFile) {
            getLog().debug("The property log.to.file is deprecated and will stop working in MUnit 2.0.0, check redirectTestOutputToFile. Please remove it from your plugin configuration.");
        }

        parseTestFilter();

        CoverageManager coverageManager = buildCoverageManager();

        try {
            StartupConfiguration startupConfiguration = createStartupConfiguration(coverageManager);
            if (startupConfiguration == null) {
                return;
            }
            if (startupConfiguration.getSuites().isEmpty()) {
                getLog().info("No MUnit suite files were found to run");
                return;
            }
            RunnerStreamConsumer streamConsumer = new RunnerStreamConsumer(buildMessageHandlers());
            ErrorStreamConsumer errorStreamConsumer = new ErrorStreamConsumer(redirectTestOutputToFile);
            JVMStarter jvmStarter = new JVMStarter(startupConfiguration, getLog());
            int result = jvmStarter.execute(streamConsumer, errorStreamConsumer);

            boolean resultSuccess;
            if (result == 0) {
                RunResult runResult = streamConsumer.getRunResult();
                coverageManager.setReport(runResult.getApplicationCoverageReport());
                coverageManager.printReport();

                for (ResultPrinter printer : getResultPrinters(startupConfiguration)) {
                    printer.print(runResult);
                }

                resultSuccess = !runResult.hasFailed();
            } else {
                throw new MojoExecutionException("Build Fail", new MojoExecutionException(errorStreamConsumer.getOutput()));
            }

            if (!resultSuccess) {
                throw new MojoExecutionException("Build Fail", new MojoExecutionException("MUnit Tests Failed"));
            }

            if (coverageManager.failBuild()) {
                throw new MojoExecutionException("Build Fail", new MojoFailureException("Coverage limits were not reached"));
            }

        } catch (MalformedURLException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        } catch (CommandLineException e) {
            e.printStackTrace();
        }
    }

    private RunnerMessageHandler buildMessageHandlers() {
        if (redirectTestOutputToFile) {
            return new TestOutputMessageHandler(getLog());
        } else {
            return new StandardOutputMessageHandler();
        }
    }

    private List<ResultPrinter> getResultPrinters(StartupConfiguration startupConfiguration) {
        List<ResultPrinter> printers = new ArrayList<ResultPrinter>();
        printers.add(new MavenResultPrinter(getLog()));
        if (enableSurefireReports) {
            printers.addAll(getSurefireResultPrinters(startupConfiguration));
        }
        if (redirectTestOutputToFile) {
            printers.add(new TestOutputPrinter(testOutputDirectory, munitTestsDirectory, project.getBasedir(), getLog()));
        }
        return printers;
    }

    private List<ResultPrinter> getSurefireResultPrinters(StartupConfiguration startupConfiguration) {
        List<ResultPrinter> printers = new ArrayList<ResultPrinter>();
        printers.add(new XmlResultPrinter(surefireReportsFolder, munitTestsDirectory, project.getBasedir(),  startupConfiguration.getSystemProperties(), getLog()));
        printers.add(new TextResultPrinter(surefireReportsFolder, munitTestsDirectory, project.getBasedir(), getLog()));
        if (redirectTestOutputToFile) {
            printers.add(new TestOutputPrinter(surefireReportsFolder, munitTestsDirectory, project.getBasedir(), getLog()));
        }
        return printers;
    }

    private StartupConfiguration createStartupConfiguration(CoverageManager coverageManager) throws MojoExecutionException, FileNotFoundException {
        if (!munitTestsDirectory.exists()) {
            getLog().warn("The project has no " + munitTestsDirectory + " folder. Aborting MUnit test run.");
            return null;
        }
        getLog().debug("MUnit root folder found at: " + munitTestsDirectory.getAbsolutePath());
        List<File> allFiles = getMunitTestSuiteFileList();

        StartupConfiguration startupConfiguration = new StartupConfiguration();
        startupConfiguration.setRunToken(UUID.randomUUID().toString());
        startupConfiguration.setBuildDirectory(project.getBuild().getDirectory());
        startupConfiguration.setClassName(REMOTE_RUNNER_CLASS.getCanonicalName());
        startupConfiguration.setClasspath(getEffectiveClassPath());
        startupConfiguration.setJvm(getEffectiveJvm());
        startupConfiguration.setWorkingDirectory(getWorkingDirectory());
        startupConfiguration.setSystemProperties(getEffectiveSystemProperties());
        startupConfiguration.setSuites(allFiles);
        startupConfiguration.setTest(testToRunName);
        startupConfiguration.setArgLines(argLines);
        startupConfiguration.setEnvironmentVariables(environmentVariables);

        coverageManager.setClasspath(startupConfiguration.getClasspath());
        startupConfiguration.setRunCoverage(coverageManager.shouldRunCoverage());
        startupConfiguration.setApplicationPaths(getConfigResources());
        startupConfiguration.setProjectName(getProjectName());
        startupConfiguration.setIgnoreFlows(coverageManager.getIgnoredFlows());

        return startupConfiguration;
    }

    private Map<String, String> getEffectiveSystemProperties() {
        SystemPropertiesManager systemPropertiesManager = new SystemPropertiesManager(
                systemPropertyVariables,
                dynamicPorts,
                session != null ? session.getUserProperties() : null,
                muleWorkingDirectory,
                getLog());
        return systemPropertiesManager.getEffectiveSystemProperties();
    }

    private List<String> getEffectiveClassPath() throws MojoExecutionException {
        ClasspathManager classpathManager = new ClasspathManager(
                classpathElements,
                getZipDependencies(),
                additionalClasspathElements,
                classpathDependencyExcludes,
                project.getArtifacts(),
                REMOTE_RUNNER_CLASS,
                getLog());
        return classpathManager.getEffectiveClasspath();
    }

    private CoverageManager buildCoverageManager() {
        CoverageManager coverageManager = new CoverageManager(coverage, munitReportsDirectory, getLog());
        getLog().debug(String.format("Coverage Manager Build for project: %s - %s", getProjectName(), project.getBasedir()));
        getLog().debug(project.getBuild().getDirectory());

        return coverageManager;
    }

    private List<File> getMunitTestSuiteFileList() throws FileNotFoundException {
        List<File> munitTestSuiteFiles = new ArrayList<File>();
        Collection<File> allFiles = FileUtils.listFiles(munitTestsDirectory, MUNIT_SUITE_EXTENSIONS, true);
        for (File file : allFiles) {
            final String fileName = file.getPath().replace(munitTestsDirectory.getPath() + File.separator, "");
            if (isValidMunitTestSuiteFile(file)) {
                if (validateFilter(fileName)) {
                    munitTestSuiteFiles.add(file);
                } else {
                    getLog().debug(String.format("MUnit Test Suite file %s skipped. It doesn't match filter criteria: [%s]", fileName, munittest));
                }
            }
        }
        return munitTestSuiteFiles;
    }

    private boolean isValidMunitTestSuiteFile(File file) throws FileNotFoundException {
        String MUNIT_TEST_SUITE_FILE_MARKER = "munit:config";
        Scanner scanner = new Scanner(file);

        while (scanner.hasNextLine()) {
            String line = scanner.nextLine();
            if (line.contains(MUNIT_TEST_SUITE_FILE_MARKER)) {
                scanner.close();
                return true;
            }
        }

        scanner.close();
        return false;
    }

    private String getConfigResources() {
        MulePropertiesLoader propertiesLoader = new MulePropertiesLoader(project.getBuild().getOutputDirectory(),
                project.getBuild().getTestOutputDirectory(),
                getLog());
        return propertiesLoader.getConfigResources();
    }

    private String getProjectName() {
        return project.getArtifactId();
    }

    private void parseTestFilter() {
        if (StringUtils.isNotBlank(munittest) && munittest.contains(SINGLE_TEST_NAME_TOKEN)) {
            testToRunName = munittest.substring(munittest.indexOf(SINGLE_TEST_NAME_TOKEN) + 1);
            munittest = munittest.substring(0, munittest.indexOf(SINGLE_TEST_NAME_TOKEN));
        } else {
            testToRunName = "";
        }
    }

    private boolean validateFilter(String fileName) {
        return munittest == null || fileName.matches(munittest);
    }


    private List<String> getZipDependencies() {
        if ("true".equals(System.getProperty(SKIP_ZIP_LOADING_PROPERTY))) {
            getLog().debug("Skipping loading of zip dependencies. Property [" + SKIP_ZIP_LOADING_PROPERTY + "] was set to true");
            return Collections.EMPTY_LIST;
        }
        List<String> zipDependencies = new ArrayList<String>();
        Set<Artifact> artifacts = project.getArtifacts();
        for (Artifact a : artifacts) {
            if (a.getFile().getPath().endsWith(".zip")) {
                getLog().debug("Loading artifact[" + a.getFile().getPath() + "]");
                zipDependencies.add(a.getFile().getPath());
            }
        }
        return zipDependencies;
    }

    private File getWorkingDirectory() {
        if (project != null) {
            return project.getBasedir();
        }
        return new File(".");
    }

    private String getEffectiveJvm() {
        String jvm = StringUtils.EMPTY;
        Toolchain toolchain = getToolchain();
        if (toolchain != null) {
            jvm = toolchain.findTool("java");
        }

        if (StringUtils.isEmpty(jvm)) {
            jvm = System.getProperty("java.home") + File.separator + "bin" + File.separator + "java";
        }
        getLog().debug("Using JVM: " + jvm);

        return jvm;
    }

    private Toolchain getToolchain() {
        Toolchain toolchain = null;
        if (toolchainManager != null && session != null) {
            toolchain = toolchainManager.getToolchainFromBuildContext("jdk", session);
        }
        return toolchain;
    }
}
