package org.mule.util;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.plugin.logging.Log;

public class ClasspathManager {

    private static final String CLASSPATH_EXCLUDE_SEPARATOR = ":";

    private List<String> zipDependencies;
    private List<String> additionalClasspathElements;
    private List<String> classpathDependencyExcludes;
    private List<String> classpathElements;

    private Set<Artifact> artifacts;
    private Class runnerClass;
    private Log log;

    public ClasspathManager(List<String> classpathElements,
                            List<String> zipDependencies,
                            List<String> additionalClasspathElements,
                            List<String> classpathDependencyExcludes,
                            Set artifacts,
                            Class runnerClass,
                            Log log) {
        this.classpathElements = classpathElements;
        this.zipDependencies = zipDependencies;
        this.additionalClasspathElements = additionalClasspathElements;
        this.classpathDependencyExcludes = classpathDependencyExcludes;
        this.artifacts = artifacts;
        this.runnerClass = runnerClass;
        this.log = log;
    }

    public List<String> getEffectiveClasspath() {
        List<String> effectiveClasspath = new ArrayList<String>();
        addClasspathElements(effectiveClasspath);
        addRunnerClassDependency(effectiveClasspath);
        addZipDependencies(effectiveClasspath);
        excludeDependenciesFromClasspath(effectiveClasspath);
        addAdditionalClasspathElements(effectiveClasspath);
        log.debug("Classpath: ");
        for (String classpathEntry : effectiveClasspath) {
            log.debug(classpathEntry);
        }
        return effectiveClasspath;
    }

    private void addClasspathElements(List<String> effectiveClasspath) {
        for(String elem : classpathElements) {
            try {
                File file = new File(elem);
                if(file.isAbsolute()) {
                    effectiveClasspath.add(file.getCanonicalPath());
                } else {
                    effectiveClasspath.add(elem);
                }
            } catch (IOException e) {
                log.error("Unable to convert classpath element to canonical form", e);
                effectiveClasspath.add(elem);
            }
        }
    }

    private void addRunnerClassDependency(List<String> effectiveClasspath) {
        if (runnerClass != null) {
            String runnerClassDependency = runnerClass.getProtectionDomain().getCodeSource().getLocation().getPath();
            effectiveClasspath.add(runnerClassDependency);
            log.debug(String.format("Runner dependency added: %s", runnerClassDependency));
        } else {
            log.info("The runner class provided is null");
        }
    }

    private void addZipDependencies(List<String> effectiveClasspath) {
        if (zipDependencies != null) {
            effectiveClasspath.addAll(zipDependencies);
        }
    }

    private void excludeDependenciesFromClasspath(List<String> effectiveClasspath) {
        if (classpathDependencyExcludes != null) {
            List<String> classpathEntriesToRemove = new ArrayList<String>();
            for (Artifact artifact : artifacts) {
                String identifier = artifact.getGroupId() + CLASSPATH_EXCLUDE_SEPARATOR + artifact.getArtifactId();
                if (classpathDependencyExcludes.contains(identifier)) {
                    String entryPath = artifact.getFile().getAbsolutePath();
                    classpathEntriesToRemove.add(entryPath);
                    log.debug(String.format("Entry %s will be removed from classpath", entryPath));
                }
            }
            effectiveClasspath.removeAll(classpathEntriesToRemove);
        }
    }

    private void addAdditionalClasspathElements(List<String> effectiveClasspath) {
        if (additionalClasspathElements != null) {
            for (String classpathEntry : additionalClasspathElements) {
                if (validateClasspathEntry(classpathEntry)) {
                    effectiveClasspath.add(classpathEntry);
                } else {
                    log.warn(String.format("Unable to add entry %s to classpath because it does not exist", classpathEntry));
                }
            }
        }
    }

    protected boolean validateClasspathEntry(String classpathEntry) {
        return new File(classpathEntry).exists();
    }

}
