package org.mule.runner;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.shared.utils.cli.CommandLineException;
import org.apache.maven.shared.utils.cli.CommandLineUtils;
import org.apache.maven.shared.utils.cli.Commandline;
import org.apache.maven.shared.utils.cli.StreamConsumer;
import org.mule.munit.remote.config.RunConfigurationBuilder;
import org.mule.munit.remote.notification.MessageHandlerMode;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;

public class JVMStarter {

    public static final String ARG_TOKEN = "-";
    private static final String SUITES_ARG = ARG_TOKEN + RunConfigurationBuilder.SUITES_PARAMETER;
    private static final String TEST_NAME_ARG = ARG_TOKEN + RunConfigurationBuilder.TEST_NAME_PARAMETER;
    private static final String RUN_TOKEN_ARG = ARG_TOKEN + RunConfigurationBuilder.RUN_TOKEN_PARAMETER;
    private static final String RUN_COVERAGE_ARG = ARG_TOKEN + RunConfigurationBuilder.RUN_COVERAGE_PARAMETER;
    private static final String PROJECT_NAME_ARG = ARG_TOKEN + RunConfigurationBuilder.PROJECT_NAME_PARAMETER;
    private static final String MODE_ARG = ARG_TOKEN + RunConfigurationBuilder.NOTIFIERS_PARAMETER;
    private static final String APPLICATION_PATHS_ARG = ARG_TOKEN + RunConfigurationBuilder.APPLICATION_PATHS_PARAMETER;
    private static final String IGNORE_FLOWS_ARG = ARG_TOKEN + RunConfigurationBuilder.IGNORE_FLOWS_PARAMETER;

    public static final String WINDOWS_OS = "windows";
    
    private static final String RUNNER_MODE = MessageHandlerMode.CONSOLE.toString();

    private static final String STARTER_CLASS_FILE = "munitstarter";
    private static final String MANIFEST_VERSION = "1.0";

    private Log log;
    private Commandline commandline = new Commandline();


    public JVMStarter(StartupConfiguration startupConfiguration, Log log) throws IOException {
        Validate.notNull(startupConfiguration);
        this.log = log;
        createJVMArgs(startupConfiguration);
        createRunnerArgs(startupConfiguration);
    }

    public int execute(StreamConsumer streamConsumer) throws CommandLineException {
        return execute(streamConsumer, streamConsumer);
    }

    public int execute(StreamConsumer outStreamConsumer, StreamConsumer errStreamConsumer) throws CommandLineException {
        log.debug("Executing command line: " + commandline);
        return CommandLineUtils.executeCommandLine(commandline, outStreamConsumer, errStreamConsumer);
    }

    private void createJVMArgs(StartupConfiguration startupConfiguration) throws IOException {
        File jarFile = createJar(startupConfiguration.getClasspath(), startupConfiguration.getClassName(), startupConfiguration.getBuildDirectory());
        commandline.setExecutable(startupConfiguration.getJvm());
        commandline.createArg().setValue("-jar");
        commandline.createArg().setValue(jarFile.getAbsolutePath());
        commandline.setWorkingDirectory(startupConfiguration.getWorkingDirectory());
        addSystemProperties(startupConfiguration);
        addArgLines(startupConfiguration);
        addEnvironmentVariables(startupConfiguration);
    }

    private void addArgLines(StartupConfiguration startupConfiguration) {
        if (startupConfiguration.getArgLines() != null) {
            for (String argLine : startupConfiguration.getArgLines()) {
                commandline.createArg(true).setValue(argLine);
            }
        }
    }

    private void addSystemProperties(StartupConfiguration startupConfig) {
        if (startupConfig.getSystemProperties() != null) {
            for (Map.Entry<String, String> sysProp : startupConfig.getSystemProperties().entrySet()) {
                commandline.createArg(true).setValue("-D" + sysProp.getKey() + "=" + sysProp.getValue());
            }
        }
    }

    private void addEnvironmentVariables(StartupConfiguration startupConfig) {
        if (startupConfig.getEnvironmentVariables() != null) {
            for (Map.Entry<String, String> envProp : startupConfig.getEnvironmentVariables().entrySet()) {
                commandline.addEnvironment(envProp.getKey(), envProp.getValue());
            }
        }
    }

    private void createRunnerArgs(StartupConfiguration startupConfiguration) {
        if (System.getProperty("os.name").toLowerCase().startsWith(WINDOWS_OS)) {
            commandline.addEnvironment(RunConfigurationBuilder.SUITES_ENV,
                    buildSuiteArgs(startupConfiguration.getSuites(), startupConfiguration.getWorkingDirectory()));
        } else {
            commandline.createArg().setValue(SUITES_ARG);
            commandline.createArg().setValue(
                    buildSuiteArgs(startupConfiguration.getSuites(), startupConfiguration.getWorkingDirectory()));
        }
        commandline.createArg().setValue(RUN_TOKEN_ARG);
        commandline.createArg().setValue(startupConfiguration.getRunToken());
        commandline.createArg().setValue(PROJECT_NAME_ARG);
        commandline.createArg().setValue(startupConfiguration.getProjectName());
        commandline.createArg().setValue(MODE_ARG);
        commandline.createArg().setValue(RUNNER_MODE);
        if (StringUtils.isNotEmpty(startupConfiguration.getTest())) {
            commandline.createArg().setValue(TEST_NAME_ARG);
            commandline.createArg().setValue(startupConfiguration.getTest());
        }
        createCoverageRunnerArgs(startupConfiguration);
    }

    private void createCoverageRunnerArgs(StartupConfiguration startupConfiguration) {
        if (startupConfiguration.isRunCoverage()) {
            commandline.createArg().setValue(RUN_COVERAGE_ARG);
            commandline.createArg().setValue(APPLICATION_PATHS_ARG);
            commandline.createArg().setValue(startupConfiguration.getApplicationPaths());
            if (!startupConfiguration.getIgnoreFlows().isEmpty()) {
                commandline.createArg().setValue(IGNORE_FLOWS_ARG);
                commandline.createArg().setValue(buildIgnoreFlowsArg(startupConfiguration.getIgnoreFlows()));
            }
        }
    }

    private String buildSuiteArgs(List<File> suites, File workingDirectory) {
        StringBuilder stringBuilder = new StringBuilder();
        String separator = StringUtils.EMPTY;
        for (File suiteFile : suites) {
            stringBuilder.append(separator);
            separator = RunConfigurationBuilder.SUITES_TOKEN_SEPARATOR;
            stringBuilder.append(workingDirectory.toURI().relativize(suiteFile.toURI()).getPath());
        }
        return stringBuilder.toString();
    }

    private String buildIgnoreFlowsArg(Set<String> ignoreFlows) {
        StringBuilder stringBuilder = new StringBuilder();
        String separator = StringUtils.EMPTY;
        for (String flow : ignoreFlows) {
            stringBuilder.append(separator);
            separator = RunConfigurationBuilder.IGNORE_FLOWS_TOKEN_SEPARATOR;
            stringBuilder.append(flow);
        }
        return stringBuilder.toString();
    }


    private File createJar(List<String> classPath, String startClassName, String buildDirectory) throws IOException {
        File tempDirectory = new File(buildDirectory);
        File file = File.createTempFile(STARTER_CLASS_FILE, ".jar", tempDirectory);
        file.deleteOnExit();
        FileOutputStream fileOutputStream = new FileOutputStream(file);
        JarOutputStream jarOutputStream = new JarOutputStream(fileOutputStream);
        try {
            jarOutputStream.setLevel(JarOutputStream.STORED);
            JarEntry je = new JarEntry("META-INF/MANIFEST.MF");
            jarOutputStream.putNextEntry(je);

            Manifest manifest = new Manifest();
            StringBuilder cp = new StringBuilder();
            for (String el : classPath) {
                cp.append(new File(el).toURI().toURL().toExternalForm()).append(" ");
            }

            manifest.getMainAttributes().putValue("Manifest-Version", MANIFEST_VERSION);
            manifest.getMainAttributes().putValue("Class-Path", cp.toString().trim());
            manifest.getMainAttributes().putValue("Main-Class", startClassName);

            manifest.write(jarOutputStream);
        } finally {
            jarOutputStream.close();
        }

        return file;
    }

}
