package org.mule.runner.handler;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.logging.Log;
import org.mule.runner.model.RunResult;
import org.mule.runner.model.SuiteResult;

import java.util.*;

public class TestOutputMessageHandler implements RunnerMessageHandler {

    private Map<String, List<String>> suiteOutputs = new HashMap<String, List<String>>();
    private List<String> suiteOutput = new ArrayList<String>();
    private String currentSuite;
    private Log log;

    public TestOutputMessageHandler(Log log) {
        this.log = log;
    }

    public void handleSuiteFinished(SuiteResult suiteResult) {
        suiteOutputs.put(suiteResult.getSuitePath(), new ArrayList<String>(suiteOutput));
        printTestSuiteFinished(suiteResult);
        currentSuite = null;
    }

    public void handleSuiteStartFailure(SuiteResult suiteResult) {
        printTestSuiteStartFailed(FilenameUtils.getName(suiteResult.getSuitePath()), suiteResult.getCause());
        suiteOutputs.put(suiteResult.getSuitePath(), Arrays.asList(suiteResult.getCause()));
        currentSuite = null;
    }

    public void handleSuiteStart(String suiteFullPath) {
        printTestSuiteStart(FilenameUtils.getName(suiteFullPath));
        suiteOutput.clear();
        currentSuite = suiteFullPath;
    }

    public void handleStandardOutputLine(String message) {
        if (shouldPrintDebugLine()) {
            System.out.println(message);
        }
        if (StringUtils.isNotBlank(currentSuite)) {
            suiteOutput.add(message);
        }
    }

    public void handleRunResult(RunResult runResult) {
        runResult.setSuiteOutputs(suiteOutputs);
    }

    public void handleSuiteFailure(SuiteResult suiteResult) {
        handleSuiteErrorOrFailure(suiteResult, "ERROR");
    }

    public void handleSuiteError(SuiteResult suiteResult) {
        handleSuiteErrorOrFailure(suiteResult, "FAILURE");
    }

    protected void handleSuiteErrorOrFailure(SuiteResult suiteResult, String tag) {
        printTestSuiteFailedOrError(FilenameUtils.getName(suiteResult.getSuitePath()), suiteResult.getCause(), tag);
        suiteOutput.add(suiteResult.getCause());
        suiteOutputs.put(suiteResult.getSuitePath(), new ArrayList<String>(suiteOutput));
        currentSuite = null;
    }

    private boolean shouldPrintDebugLine() {
        return currentSuite == null && log.isDebugEnabled();
    }

    private void printTestSuiteStart(String suiteName) {
        System.out.format("Running %s\n", suiteName);
    }

    private void printTestSuiteFinished(SuiteResult suiteResult) {
        System.out.format("Tests run: %d, Failures: %d, Errors: %d, Skipped: %d, Time elapsed: %.3f sec\n",
                suiteResult.getNumberOfTests(),
                suiteResult.getNumberOfFailures(),
                suiteResult.getNumberOfErrors(),
                suiteResult.getNumberOfIgnores(),
                suiteResult.getTime() / 1000.0f);
    }

    private void printTestSuiteStartFailed(String suiteName, String cause) {
        System.out.format("%s failed to start\n", suiteName);
        System.out.println(cause);
    }

    private void printTestSuiteFailedOrError(String suiteName, String cause, String tag) {
        System.out.format("%s failed <<< %s\n", suiteName, tag);
        System.out.println(cause);
    }
}
