package org.mule.runner;


import org.apache.maven.shared.utils.cli.StreamConsumer;
import org.mule.runner.handler.RunnerMessageHandler;
import org.mule.runner.model.RunResult;

public class RunnerStreamConsumer implements StreamConsumer {

    public static final String FIELD_TOKEN = new String(new byte[]{0x1b});
    public static final String NEW_LINE_ESCAPE_TOKEN = new String(new byte[]{0x1a});

    public static final String RUN_START_MSG_ID = "0";
    public static final String TEST_SUITE_START_FAILURE_MSG_ID = "1";
    public static final String TEST_SUITE_START_MSG_ID = "2";
    public static final String NUMBER_OF_TESTS_MSG_ID = "3";
    public static final String TEST_START_MSG_ID = "4";
    public static final String TEST_FINISHED_MSG_ID = "5";
    public static final String TEST_IGNORED_MSG_ID = "6";
    public static final String TEST_FAILURE_MSG_ID = "7";
    public static final String TEST_ERROR_MSG_ID = "8";
    public static final String TEST_SUITE_FINISHED_MSG_ID = "9";
    public static final String RUN_FINISHED_MSG_ID = "10";
    public static final String COVERAGE_REPORT_MSG_ID = "11";
    public static final String BEFORE_SUITE_FAILURE = "12";
    public static final String BEFORE_SUITE_ERROR = "13";
    public static final String AFTER_SUITE_FAILURE = "14";
    public static final String AFTER_SUITE_ERROR = "15";


    private RunResult runResult = new RunResult();
    private RunnerMessageHandler runnerMessageHandler;

    public RunnerStreamConsumer(RunnerMessageHandler runnerMessageHandler) {
        this.runnerMessageHandler = runnerMessageHandler;
    }

    public void consumeLine(String s) {
        if (s.startsWith(FIELD_TOKEN)) {
            handleProtocolLine(s);
        } else {
            runnerMessageHandler.handleStandardOutputLine(s);
        }
    }

    protected void handleProtocolLine(String message) {
        RunnerMessage msg = new RunnerMessage(message, FIELD_TOKEN, NEW_LINE_ESCAPE_TOKEN);
        String suiteFullPath = msg.getSuiteFullPath();
        if (msg.getMessageId().equals(RUN_START_MSG_ID)) {
            runResult.runStarted(msg.getRunToken());
        }
        if (msg.getMessageId().equals(TEST_SUITE_START_FAILURE_MSG_ID)) {
            runResult.testSuiteStartFailed(suiteFullPath, msg.getStackTrace());
            runnerMessageHandler.handleSuiteStartFailure(runResult.getSuite(suiteFullPath));
        }
        if (msg.getMessageId().equals(TEST_SUITE_START_MSG_ID)) {
            runResult.testSuiteStarted(suiteFullPath);
            runnerMessageHandler.handleSuiteStart(suiteFullPath);
        }
        if (msg.getMessageId().equals(TEST_START_MSG_ID)) {
            runResult.testStarted(suiteFullPath, msg.getTestName());
        }
        if (msg.getMessageId().equals(NUMBER_OF_TESTS_MSG_ID)) {
            runResult.setNumberOfTests(suiteFullPath, Integer.valueOf(msg.getFreeMessage()));
        }
        if (msg.getMessageId().equals(TEST_FINISHED_MSG_ID)) {
            runResult.testFinished(suiteFullPath, msg.getTestName());
        }
        if (msg.getMessageId().equals(TEST_IGNORED_MSG_ID)) {
            runResult.testIgnored(suiteFullPath, msg.getTestName());
        }
        if (msg.getMessageId().equals(TEST_FAILURE_MSG_ID)) {
            runResult.testFailure(suiteFullPath, msg.getTestName(), msg.getStackTrace());
        }
        if (msg.getMessageId().equals(TEST_ERROR_MSG_ID)) {
            runResult.testError(suiteFullPath, msg.getTestName(), msg.getStackTrace());
        }
        if (msg.getMessageId().equals(TEST_SUITE_FINISHED_MSG_ID)) {
            runResult.suiteFinished(suiteFullPath);
            runnerMessageHandler.handleSuiteFinished(runResult.getSuite(suiteFullPath));
        }
        if (msg.getMessageId().equals(BEFORE_SUITE_FAILURE) ||
                msg.getMessageId().equals(AFTER_SUITE_FAILURE)) {
            runResult.suiteFailure(suiteFullPath, msg.getStackTrace());
            runnerMessageHandler.handleSuiteFailure(runResult.getSuite(suiteFullPath));
        }
        if (msg.getMessageId().equals(BEFORE_SUITE_ERROR) ||
                msg.getMessageId().equals(AFTER_SUITE_ERROR)) {
            runResult.suiteError(suiteFullPath, msg.getStackTrace());
            runnerMessageHandler.handleSuiteError(runResult.getSuite(suiteFullPath));
        }
        if (msg.getMessageId().equals(COVERAGE_REPORT_MSG_ID)) {
            runResult.setCoverageReport(msg.getJsonMessage());
        }
        if (msg.getMessageId().equals(RUN_FINISHED_MSG_ID)) {
            runResult.setRunFinished(true);
        }
    }

    public RunResult getRunResult() {
        runnerMessageHandler.handleRunResult(runResult);
        return runResult;
    }

}
