package org.mule.runner.model;

import com.google.gson.Gson;
import org.mule.munit.plugins.coverage.report.ApplicationCoverageReport;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class RunResult {

    private String token;

    private SuiteResult currentSuite = null;

    private long currentSuiteTime;
    private long currentTestTime;

    private ApplicationCoverageReport coverageReport;
    private Map<String, List<String>> suiteOutputs;

    private boolean runFinished = false;
    private boolean coverageEnabled = true;

    private Map<String, SuiteResult> suites = new HashMap<String, SuiteResult>();

    public void runStarted(String token) {
        this.token = token;
    }

    public void testSuiteStartFailed(String suiteFullPath, String stackTrace) {
        SuiteResult suite = new SuiteResult();
        suite.setRuntimeStartFailed(true);
        suite.setCause(stackTrace);
        suite.setSuiteFinished(true);
        suite.setSuitePath(suiteFullPath);
        suites.put(suiteFullPath, suite);
        suite.setTime(0);
    }

    public void testSuiteStarted(String suiteFullPath) {
        SuiteResult suite = new SuiteResult();
        suite.setSuitePath(suiteFullPath);
        suites.put(suiteFullPath, suite);
        currentSuite = suite;
        currentSuiteTime = System.currentTimeMillis();
    }

    public void testStarted(String suiteFullPath, String testName) {
        suites.get(suiteFullPath).add(testName, new TestResult(testName));
        currentTestTime = System.currentTimeMillis();
    }

    public void setNumberOfTests(String suiteFullPath, Integer numberTests) {
        suites.get(suiteFullPath).setNumberOfTests(numberTests);
    }

    public void testFailure(String suiteFullPath, String testName, String stackTrace) {
        TestResult test = suites.get(suiteFullPath).getTest(testName);
        test.setFailed(true);
        test.setCause(stackTrace);
        test.setTime(System.currentTimeMillis() - currentTestTime);
    }

    public void testError(String suiteFullPath, String testName, String stackTrace) {
        TestResult test = suites.get(suiteFullPath).getTest(testName);
        test.setError(true);
        test.setCause(stackTrace);
        test.setTime(System.currentTimeMillis() - currentTestTime);
    }

    public void testFinished(String suiteFullPath, String testName) {
        TestResult test = suites.get(suiteFullPath).getTest(testName);
        test.setFinished(true);
        test.setTime(System.currentTimeMillis() - currentTestTime);
    }

    public void testIgnored(String suiteFullPath, String testName) {
        TestResult test = suites.get(suiteFullPath).getTest(testName);
        test.setIgnored(true);
        test.setTime(System.currentTimeMillis() - currentTestTime);
    }

    public void suiteFinished(String suiteFullPath) {
        suites.get(suiteFullPath).setSuiteFinished(true);
        currentSuite = null;
        suites.get(suiteFullPath).setTime(System.currentTimeMillis() - currentSuiteTime);
    }

    public void suiteFailure(String suiteFullPath, String stackTrace) {
        suites.get(suiteFullPath).setSuiteFailed(true);
        suites.get(suiteFullPath).setCause(stackTrace);
        currentSuite = null;
        suites.get(suiteFullPath).setTime(System.currentTimeMillis() - currentSuiteTime);
    }

    public void suiteError(String suiteFullPath, String stackTrace) {
        suites.get(suiteFullPath).setSuiteError(true);
        suites.get(suiteFullPath).setCause(stackTrace);
        currentSuite = null;
        suites.get(suiteFullPath).setTime(System.currentTimeMillis() - currentSuiteTime);
    }

    public Collection<SuiteResult> getSuites() {
        return suites.values();
    }

    public boolean isRunFinished() {
        return runFinished;
    }

    public void setRunFinished(boolean runFinished) {
        this.runFinished = runFinished;
    }

    public SuiteResult getSuite(String suiteName) {
        return suites.get(suiteName);
    }

    public int getNumberOfTests() {
        int tests = 0;
        for (SuiteResult suite : getSuites()) {
            tests += suite.getNumberOfTests();
        }
        return tests;
    }

    public int getProcessedTests() {
        int processedTests = 0;
        for (SuiteResult suite : getSuites()) {
            processedTests += suite.getNumberOfProcessedTests();
        }
        return processedTests;
    }

    public int getNumberOfIgnores() {
        int ignores = 0;
        for (SuiteResult suite : getSuites()) {
            ignores += suite.getNumberOfIgnores();
        }
        return ignores;
    }

    public int getNumberOfErrors() {
        int errors = 0;
        for (SuiteResult suite : getSuites()) {
            errors += suite.getNumberOfErrors();
        }
        return errors;
    }

    public int getNumberOfFailures() {
        int failures = 0;
        for (SuiteResult suite : getSuites()) {
            failures += suite.getNumberOfFailures();
        }
        return failures;
    }

    public String getToken() {
        return token;
    }

    public SuiteResult getCurrentSuite() {
        return currentSuite;
    }

    public ApplicationCoverageReport getApplicationCoverageReport() {
        return coverageReport;
    }

    public void setCoverageReport(String jsonReport) {
        this.coverageReport = new Gson().fromJson(jsonReport, ApplicationCoverageReport.class);
    }

    public Map<String, List<String>> getSuiteOutputs() {
        return suiteOutputs;
    }

    public void setSuiteOutputs(Map<String, List<String>> suiteOutputs) {
        this.suiteOutputs = suiteOutputs;
    }

    public boolean hasFailed() {
        for (SuiteResult suite : suites.values()) {
            if (suite.hasFailed() || suite.hasError()) {
                return true;
            }
        }
        return false;
    }

    public void setCoverageEnabled(boolean runCoverage) {
        this.coverageEnabled = runCoverage;
    }

    public boolean isCoverageEnabled() {
        return coverageEnabled;
    }

}
