package org.mule.util;

import org.apache.maven.plugin.logging.Log;
import org.mule.api.config.MuleProperties;
import org.mule.munit.common.util.FreePortFinder;

import java.io.File;
import java.io.IOException;
import java.util.*;

import static java.util.Arrays.asList;

public class SystemPropertiesManager {

    private static final int MIN_PORT_NUMBER = 40000;
    private static final int MAX_PORT_NUMBER = 50000;
    private static final Collection<String> INVALID_SYSTEM_PROPERTIES =
            asList("java.library.path", "file.encoding", "jdk.map.althashing.threshold");

    private List<String> dynamicPorts;
    private Map<String, String> systemPropertyVariables;
    private Properties userProperties;
    private File muleWorkingDir;
    private Log log;

    public SystemPropertiesManager(Map<String, String> systemPropertyVariables, List<String> dynamicPorts, Properties userProperties, File muleWorkingDir, Log log) {
        this.systemPropertyVariables = systemPropertyVariables;
        this.dynamicPorts = dynamicPorts;
        this.userProperties = userProperties;
        this.muleWorkingDir = muleWorkingDir;
        this.log = log;
    }

    public Map<String, String> getEffectiveSystemProperties() {
        Map<String, String> effectiveSystemProperties = new HashMap<String, String>();
        setSystemPropertyVariables(effectiveSystemProperties);
        setDynamicPorts(effectiveSystemProperties);
        stopLicenseCheck(effectiveSystemProperties);
        setMuleWorkingDir(effectiveSystemProperties);
        setUserSystemProperties(effectiveSystemProperties);
        removeInvalidSystemProperties(effectiveSystemProperties);
        return effectiveSystemProperties;
    }

    private void stopLicenseCheck(Map<String, String> props) {
        log.debug("Avoid license check for Mule EE components...");
        props.put("mule.testingMode", "true");
    }

    private void setMuleWorkingDir(Map<String, String> props) {
        if (muleWorkingDir != null) {
            try {
                log.info("Mule working directory set to: " + muleWorkingDir);
                props.put(MuleProperties.MULE_WORKING_DIRECTORY_PROPERTY, muleWorkingDir.getCanonicalPath());
            } catch (IOException e) {
                log.debug("Unable to set the Mule working directory", e);
            }
        }

    }


    private void setSystemPropertyVariables(Map<String, String> systemProperties) {
        if (systemPropertyVariables != null) {
            systemProperties.putAll(systemPropertyVariables);
            log.debug("Adding System Property Variables : " + systemPropertyVariables);
        }
    }

    private void setUserSystemProperties(Map<String, String> systemProperties) {
        if (userProperties != null) {
            for (Map.Entry<Object, Object> prop : userProperties.entrySet()) {
                Object key = prop.getKey();
                Object value = prop.getValue();
                if (key instanceof String && value instanceof String) {
                    systemProperties.put(prop.getKey().toString(), prop.getValue().toString());
                    log.debug(String.format("Setting System Property [%s] to %s", key, value));
                }
            }
        }
    }

    private void setDynamicPorts(Map<String, String> systemProperties) {
        if (dynamicPorts != null) {
            log.info("Acquiring dynamic ports...");
            FreePortFinder portFinder = new FreePortFinder(MIN_PORT_NUMBER, MAX_PORT_NUMBER);
            for (String portPlaceHolder : dynamicPorts) {
                Integer dynamicPort = portFinder.find();
                systemProperties.put(portPlaceHolder, dynamicPort.toString());
                log.debug(String.format("Dynamic port [%s] set to: [%s]", portPlaceHolder, dynamicPort));
            }
            log.info("Dynamic port definition [DONE]");
        }
    }

    private void removeInvalidSystemProperties(Map<String, String> effectiveSystemProperties) {
        for (String invalidProp : INVALID_SYSTEM_PROPERTIES) {
            if (effectiveSystemProperties.containsKey(invalidProp)) {
                effectiveSystemProperties.remove(invalidProp);
                log.warn(invalidProp + " cannot be set as system property, use <argLine>-D"
                        + invalidProp + "=...</argLine> instead");
            }
        }
    }
}
