/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit;

import static org.mule.munit.remote.FolderNames.APPLICATION;
import static org.mule.munit.remote.FolderNames.META_INF;
import static org.mule.munit.remote.FolderNames.MULE;
import static org.mule.munit.remote.FolderNames.MULE_ARTIFACT;
import static org.mule.munit.remote.FolderNames.MUNIT;
import static org.mule.munit.remote.FolderNames.MUNIT_WORKING_DIR;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.maven.execution.MavenSession;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;
import org.apache.maven.project.MavenProject;
import org.apache.maven.shared.utils.cli.CommandLineException;
import org.apache.maven.toolchain.ToolchainManager;

import org.mule.coverage.CoverageManager;
import org.mule.munit.common.util.FileUtils;
import org.mule.munit.mojo.JVMLocator;
import org.mule.munit.mojo.MessageHandlerFactory;
import org.mule.munit.mojo.ResultPrinterFactory;
import org.mule.munit.mojo.TestSuiteFileFilter;
import org.mule.munit.mojo.exceptions.MojoExecutionExceptionFactory;
import org.mule.munit.remote.ApplicationStructureGenerator;
import org.mule.munit.remote.RemoteRunner;
import org.mule.munit.remote.config.RunConfiguration;
import org.mule.munit.remote.config.RunConfigurationParser;
import org.mule.munit.util.JarFileFactory;
import org.mule.munit.util.RunConfigurationFactory;
import org.mule.runner.JVMStarter;
import org.mule.runner.consumer.ErrorStreamConsumer;
import org.mule.runner.consumer.RunnerStreamConsumer;
import org.mule.runner.model.RunResult;
import org.mule.runner.printer.ResultPrinter;
import org.mule.util.ArgLinesManager;
import org.mule.util.ClasspathManager;
import org.mule.util.properties.UserPropertiesBuilder;

import com.google.gson.GsonBuilder;


@Mojo(name = "test",
    defaultPhase = LifecyclePhase.TEST,
    requiresDependencyResolution = ResolutionScope.TEST)
public class MUnitMojo extends AbstractMojo {

  public static final String SINGLE_TEST_NAME_TOKEN = "#";
  public static final String SITE_MUNIT_DIRECTORY = "munit";

  private static final String SKIP_TESTS_PROPERTY = "skipTests";
  private static final String SKIP_MUNIT_TESTS_PROPERTY = "skipMunitTests";
  private static final String SKIP_ZIP_LOADING_PROPERTY = "skipZipLoading";

  private static final String ARG_TOKEN = "-";
  public static final String STARTER_CLASS_FILE = "munitstarter";
  private static final Class REMOTE_RUNNER_CLASS = RemoteRunner.class;
  private static final String RUN_CONFIGURATION_ARG = ARG_TOKEN + RunConfigurationParser.RUN_CONFIGURATION_PARAMETER;
  public static final String RUN_CONFIGURATION_JSON = "run-configuration.json";

  public static final String POM_XML_FILE_NAME = "pom.xml";
  public static final String MULE_APPLICATION_JSON_FILE_NAME = "mule-application.json";

  @Parameter(property = "project", required = true)
  protected MavenProject project;

  @Parameter(property = "project.testClasspathElements", required = true, readonly = true)
  protected List<String> classpathElements;

  @Parameter(property = "munit.test")
  protected String munitTest;

  @Parameter(property = "munit.tags")
  protected String munitTags;

  @Parameter(defaultValue = "${skipMunitTests}")
  protected boolean skipMunitTests = false;

  @Parameter(property = "system.property.variables")
  protected Map<String, String> systemPropertyVariables;

  @Parameter(property = "environment.variables")
  protected Map<String, String> environmentVariables;

  @Parameter(property = "dynamic.ports")
  protected List<String> dynamicPorts;

  @Parameter(property = "munit.coverage")
  protected Coverage coverage;

  @Parameter(defaultValue = "${munit.randomFailMessages}")
  private boolean randomFailMessages = false;

  @Parameter(property = "argLines")
  protected List<String> argLines;

  @Parameter(property = "additionalClasspathElements")
  protected List<String> additionalClasspathElements;

  @Parameter(property = "sharedLibraries")
  protected List<String> sharedLibraries;

  @Parameter(property = "classpathDependencyExcludes")
  protected List<String> classpathDependencyExcludes;

  @Parameter(defaultValue = "${project.build.directory}/test-mule/munit")
  protected File munitTestsDirectory;

  @Parameter(defaultValue = "${project.build.directory}/munit-reports/")
  protected File munitReportsDirectory;

  @Parameter(defaultValue = "${project.build.directory}/site/")
  protected File projectSiteDirectory;

  @Parameter(property = "redirectTestOutputToFile", defaultValue = "false")
  protected boolean redirectTestOutputToFile;

  @Parameter(defaultValue = "${project.build.directory}/munit-reports/output/")
  protected File testOutputDirectory;

  @Parameter(property = "enableSurefireReports", defaultValue = "true")
  protected boolean enableSurefireReports;

  @Parameter(defaultValue = "${project.build.directory}/surefire-reports/")
  protected File surefireReportsFolder;

  @Parameter
  protected String runtimeVersion;

  @Parameter(property = "runtimeProduct", defaultValue = "MULE_EE")
  protected String runtimeProduct;

  @Component
  protected MavenSession session;

  @Component
  protected ToolchainManager toolchainManager;

  private File munitWorkingDirectory;

  protected TestSuiteFileFilter testSuiteFileFilter;
  protected MojoExecutionExceptionFactory exceptionFactory;

  protected ClasspathManager classpathManager;
  protected CoverageManager coverageManager;
  protected ResultPrinterFactory resultPrinterFactory;
  protected MessageHandlerFactory messageHandlerFactory;
  protected RunConfigurationFactory runConfigurationFactory;
  protected JVMLocator jvmLocator;
  protected JarFileFactory jarFileFactory;
  protected Map<String, String> effectiveSystemProperties;

  public void execute() throws MojoExecutionException {
    if (!"true".equals(System.getProperty(SKIP_TESTS_PROPERTY))) {
      if (!skipMunitTests) {
        init();
        doExecute();
      } else {
        getLog().info("Run of munit-maven-plugin skipped. Property [" + SKIP_MUNIT_TESTS_PROPERTY + "] was set to true");
      }
    } else {
      getLog().info("Run of munit-maven-plugin skipped. Property [" + SKIP_TESTS_PROPERTY + "] was set to true");
    }
  }

  protected void init() throws MojoExecutionException {
    this.exceptionFactory = new MojoExecutionExceptionFactory(randomFailMessages);
    this.testSuiteFileFilter = new TestSuiteFileFilter(getLog(), munitTest);
    this.messageHandlerFactory = new MessageHandlerFactory(getLog());
    this.effectiveSystemProperties = getEffectiveSystemProperties();
    this.resultPrinterFactory = new ResultPrinterFactory(testOutputDirectory, surefireReportsFolder,
                                                         effectiveSystemProperties, getLog());

    this.munitWorkingDirectory = createAndGetMunitWorkingDirectory();
    File applicationJsonFile =
        Paths.get(project.getBuild().getDirectory(), META_INF.value(), MULE_ARTIFACT.value(), MULE_APPLICATION_JSON_FILE_NAME)
            .toFile();
    File workingDirMunitSrcFolder =
        Paths.get(munitWorkingDirectory.getAbsolutePath(), APPLICATION.value(), MULE.value(), MUNIT.value()).toFile();
    this.runConfigurationFactory =
        new RunConfigurationFactory(getLog(), munitTags, runtimeVersion, runtimeProduct, munitWorkingDirectory,
                                    testSuiteFileFilter, project, session, workingDirMunitSrcFolder, applicationJsonFile);

    this.jvmLocator = new JVMLocator(session, toolchainManager, getLog());
    this.classpathManager = new ClasspathManager(classpathElements,
                                                 additionalClasspathElements,
                                                 project.getArtifacts(),
                                                 REMOTE_RUNNER_CLASS,
                                                 getLog());

    this.jarFileFactory = new JarFileFactory();
    this.coverageManager = buildCoverageManager();
  }

  protected void doExecute() throws MojoExecutionException {
    generateApplicationStructure();

    RunConfiguration runConfiguration = runConfigurationFactory.createRunConfiguration(coverageManager);
    try {

      if (runConfiguration == null) {
        getLog().info("No Run configuration created not running MUnit ");
        return;
      }
      if (runConfiguration.getSuitePaths().isEmpty()) {
        getLog().info("No MUnit suite files were found to run");
        return;
      }

      JVMStarter jvmStarter = createJVMStarter(runConfiguration);
      RunnerStreamConsumer streamConsumer = new RunnerStreamConsumer(messageHandlerFactory.create(redirectTestOutputToFile));
      ErrorStreamConsumer errorStreamConsumer = new ErrorStreamConsumer(redirectTestOutputToFile);
      int result = jvmStarter.execute(streamConsumer, errorStreamConsumer);

      boolean resultSuccess;
      String stackTrace;
      if (result == 0) {
        RunResult runResult = streamConsumer.getRunResult();

        // TODO disable due to MU-1091
        // coverageManager.setReport(runResult.getApplicationCoverageReport());
        // coverageManager.printReport();

        for (ResultPrinter printer : resultPrinterFactory.create(enableSurefireReports, redirectTestOutputToFile)) {
          printer.print(runResult);
        }

        resultSuccess = !runResult.hasFailed();
        stackTrace = runResult.getStackTrace();
      } else {
        throw new MojoExecutionException("Build Fail", new MojoExecutionException(errorStreamConsumer.getOutput()));
      }

      if (!resultSuccess) {
        throw exceptionFactory.buildException("Build Fail", new MojoExecutionException("MUnit Tests Failed\n" + stackTrace));
      }

      if (coverageManager.failBuild()) {
        throw new MojoExecutionException("Build Fail", new MojoFailureException("Coverage limits were not reached"));
      }

    } catch (IOException | CommandLineException e) {
      e.printStackTrace();
    }
  }

  private File createAndGetMunitWorkingDirectory() {
    File workingDirectory =
        new File(project.getBuild().getDirectory(), MUNIT_WORKING_DIR.value() + "-" + Long.toString(System.nanoTime()));
    workingDirectory.mkdir();
    return workingDirectory;
  }

  protected void generateApplicationStructure() throws MojoExecutionException {
    File sourceFolder = Paths.get(project.getBuild().getDirectory()).toFile();
    try {
      getLog().debug("Attempting to create application structure source: " + sourceFolder.getAbsolutePath() + " - destination: "
          + munitWorkingDirectory.getAbsolutePath());
      new ApplicationStructureGenerator(sourceFolder, munitWorkingDirectory).generate();
    } catch (Exception e) {
      throw new MojoExecutionException("Fail to create application structure", e);
    }
  }

  protected JVMStarter createJVMStarter(RunConfiguration runConfiguration)
      throws MojoExecutionException, IOException {
    if (!munitTestsDirectory.exists()) {
      getLog().warn("The project has no " + munitTestsDirectory + " folder. Aborting MUnit test run.");
      return null;
    }
    getLog().debug("MUnit root folder found at: " + munitTestsDirectory.getAbsolutePath());

    String starterJarFileName = FileUtils.generateRandomFileName(STARTER_CLASS_FILE, ".jar");

    File jarFile = jarFileFactory.create(classpathManager.getEffectiveClasspath(), REMOTE_RUNNER_CLASS.getCanonicalName(),
                                         new File(project.getBuild().getDirectory()), starterJarFileName);

    JVMStarter jvmStarter = new JVMStarter(getLog())
        .withJVM(jvmLocator.locate())
        .withWorkingDirectory(getWorkingDirectory())
        .withJar(jarFile)
        .withArgLines(getEffectiveArgLines(starterJarFileName))
        .withSystemProperties(effectiveSystemProperties)
        .addEnvironmentVariables(environmentVariables);

    Map<String, File> fileArgLines = new HashMap<>();
    fileArgLines.put(RUN_CONFIGURATION_ARG, saveRunConfigurationToFile(runConfiguration));
    jvmStarter.withArgLines(fileArgLines);

    return jvmStarter;
  }

  protected File saveRunConfigurationToFile(RunConfiguration runConfiguration) throws IOException {
    File runConfigurationFile =
        Paths.get(runConfiguration.getContainerConfiguration().getMunitWorkingDirectoryPath(), RUN_CONFIGURATION_JSON)
            .toFile();
    FileWriter fileWriter = new FileWriter(runConfigurationFile);
    fileWriter.write(new GsonBuilder().setPrettyPrinting().create().toJson(runConfiguration));
    fileWriter.flush();
    fileWriter.close();

    return runConfigurationFile;
  }

  protected List<String> getEffectiveArgLines(String starterJarFileName) {
    return new ArgLinesManager(argLines, starterJarFileName, getLog()).getEffectiveArgLines();
  }

  protected CoverageManager buildCoverageManager() {
    File mavenSiteFolder = getOrCreateSiteMunitDirectory();

    CoverageManager coverageManager = new CoverageManager(coverage, mavenSiteFolder, munitReportsDirectory, getLog());
    getLog().debug(String.format("Coverage Manager Build for project: %s - %s", project.getArtifactId(), project.getBasedir()));
    getLog().debug(project.getBuild().getDirectory());

    return coverageManager;
  }

  protected File getWorkingDirectory() {
    if (project != null) {
      return project.getBasedir();
    }
    return new File(".");
  }

  private Map<String, String> getEffectiveSystemProperties() {
    Map<String, String> effectiveSystemProperties = new HashMap<>();
    effectiveSystemProperties.putAll(getUserSystemProperties());
    // TODO MU-978
    effectiveSystemProperties.put("org.glassfish.grizzly.DEFAULT_MEMORY_MANAGER",
                                  "org.glassfish.grizzly.memory.HeapMemoryManager");
    return effectiveSystemProperties;
  }

  private Map<String, String> getUserSystemProperties() {
    return new UserPropertiesBuilder(project.getBuild().getDirectory(), getLog())
        .withSystemPropertyVariables(systemPropertyVariables)
        .withDynamicPorts(dynamicPorts)
        .withUserProperties(session != null ? session.getUserProperties() : null)
        .build();
  }

  private File getOrCreateSiteMunitDirectory() {
    if (!projectSiteDirectory.exists()) {
      projectSiteDirectory.mkdir();
    }

    File munitSiteDirectory = new File(projectSiteDirectory, SITE_MUNIT_DIRECTORY);
    if (!munitSiteDirectory.exists()) {
      munitSiteDirectory.mkdir();
    }

    return munitSiteDirectory;
  }
}
