/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.mojo.locators;

import org.apache.maven.plugin.logging.Log;

import java.io.File;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;

/**
 * It locates LOG4J2 configuration files in a given directory.
 * 
 * @author Mulesoft Inc.
 * @since 2.0.0
 */
public class Log4J2ConfigurationLocator implements FilesLocator {

  private static final String CLASSES_DIRECTORY = "classes";
  private static final String TEST_CLASSES_DIRECTORY = "test-classes";

  private static final String LOG4J2__FILE_NAME = "log4j2.xml";
  private static final String LOG4J2_TEST_FILE_NAME = "log4j2-test.xml";

  private Log log;

  public Log4J2ConfigurationLocator(Log log) {
    this.log = log;
  }

  /**
   * It locates log4j2 configuration files starting in the rootPath sent.
   *
   * Resolution Order:
   *
   * 1. A file called {@code log4j2-test.xml} will be searched
   *
   * 2. A file called {@code log4j2.xml}
   *
   * @param folder path to use in order to locateFiles the configuration files
   * @return a list with the located files where the first one is the more representative, empty list if no files where located
   */
  @Override
  public List<File> locateFiles(File folder) {
    Path rootPath = Paths.get(folder.toURI());
    List<File> log4JFiles = new ArrayList<>();

    Path configurationFilePath = rootPath.resolve(TEST_CLASSES_DIRECTORY).resolve(LOG4J2_TEST_FILE_NAME);

    if (configurationFilePath.toFile().exists()) {
      log.debug("Located file at: " + configurationFilePath.toAbsolutePath().toString());
      log4JFiles.add(configurationFilePath.toFile());
    }

    configurationFilePath = rootPath.resolve(CLASSES_DIRECTORY).resolve(LOG4J2__FILE_NAME);
    if (configurationFilePath.toFile().exists()) {
      log.debug("Located file at: " + configurationFilePath.toAbsolutePath().toString());

      log4JFiles.add(configurationFilePath.toFile());
    }
    if (log4JFiles.isEmpty()) {
      log.debug("No file located");
    }

    return log4JFiles;
  }


}
