/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.util;

import static com.google.common.collect.Sets.newHashSet;
import static org.apache.commons.lang3.StringUtils.isBlank;

import java.io.File;
import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.execution.MavenSession;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.logging.Log;
import org.apache.maven.project.MavenProject;
import org.mule.coverage.CoverageManager;
import org.mule.munit.common.util.FileUtils;
import org.mule.munit.mojo.TestSuiteFileFilter;
import org.mule.munit.mojo.locators.Log4J2ConfigurationLocator;
import org.mule.munit.mojo.locators.TestSuiteFilesLocator;
import org.mule.munit.remote.config.*;
import org.mule.munit.remote.notifiers.ConsoleNotifier;
import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.api.deployment.persistence.MuleApplicationModelJsonSerializer;

public class RunConfigurationFactory {

  private Log log;
  private String munitTags;
  private String testToRunName;
  private String runtimeVersion;
  private String runtimeProduct;
  private File munitWorkingDirectory;
  private TestSuiteFileFilter testSuiteFileFilter;

  protected MavenProject project;

  protected MavenSession session;

  private File munitSrcFolder;
  private File applicationJsonFile;

  public RunConfigurationFactory(Log log, String munitTags, String runtimeVersion, String runtimeProduct,
                                 File munitWorkingDirectory, TestSuiteFileFilter testSuiteFileFilter, MavenProject project,
                                 MavenSession session, File munitSrcFolder, File applicationJsonFile) {
    this.log = log;
    this.munitTags = munitTags;
    this.runtimeVersion = runtimeVersion;
    this.runtimeProduct = runtimeProduct;
    this.munitWorkingDirectory = munitWorkingDirectory;
    this.project = project;
    this.session = session;
    this.testSuiteFileFilter = testSuiteFileFilter;
    this.testToRunName = testSuiteFileFilter.getTestNameRegEx();

    this.munitSrcFolder = munitSrcFolder;
    this.applicationJsonFile = applicationJsonFile;
  }

  public RunConfiguration createRunConfiguration(CoverageManager coverageManager) throws MojoExecutionException {
    String runToken = UUID.randomUUID().toString();

    CoverageConfiguration coverageConfiguration = new CoverageConfiguration.CoverageConfigurationBuilder()
        .withShouldRunCoverage(coverageManager.shouldRunCoverage())
        .withApplicationPaths(getConfigResources())
        .withIgnoredFlowNames(coverageManager.getIgnoredFlows())
        .buid();

    List<NotifierParameter> consoleParameters = new ArrayList<>();
    consoleParameters.add(new NotifierParameter("runToken", String.class.getCanonicalName(), runToken));
    NotifierConfiguration consoleNotifierConfiguration = new NotifierConfiguration.NotifierConfigurationBuilder()
        .withClazz(ConsoleNotifier.class.getCanonicalName())
        .withParameters(consoleParameters)
        .build();

    List<NotifierConfiguration> notifierConfigurations = new ArrayList<>();
    notifierConfigurations.add(consoleNotifierConfiguration);

    // TODO find out what is a global settings
    // TODO set force update as maven parameter
    MavenConfiguration mavenConfiguration = new MavenConfiguration.MavenConfigurationBuilder()
        .withMavenRepositoryDirectoryPath(session.getLocalRepository().getBasedir())
        .withSettingsXmlFilePath(session.getRequest().getUserSettingsFile().getAbsolutePath())
        .withForcePolicyUpdate(true)
        .build();

    ContainerConfiguration containerConfiguration = new ContainerConfiguration.ContainerConfigurationBuilder()
        .withRuntimeId(getRuntimeVersion())
        .withProduct(runtimeProduct)
        .withMunitWorkingDirectoryPath(munitWorkingDirectory.getAbsolutePath())
        .withLog4JConfigurationFilePath(getLog4JConfigurationFilePath())
        .withMavenConfiguration(mavenConfiguration)
        .build();

    RunConfiguration.RunConfigurationBuilder builder = new RunConfiguration.RunConfigurationBuilder();

    builder.withRunToken(runToken)
        .withProjectName(project.getName())
        .withTags(munitTags == null ? Collections.emptySet() : new HashSet<>(Arrays.asList(munitTags.split(","))))
        .withTestNames(isBlank(testToRunName) ? Collections.emptySet() : newHashSet(testToRunName.split(",")))
        .withSuitePaths(locateMunitTestSuites())
        .withNotifierConfigurations(notifierConfigurations)
        .withCoverageConfiguration(coverageConfiguration)
        .withContainerConfiguration(containerConfiguration);

    return builder.build();
  }

  private String getLog4JConfigurationFilePath() {
    return getLog4JConfigurationFile().isPresent() ? getLog4JConfigurationFile().get().getAbsolutePath() : StringUtils.EMPTY;
  }

  private Optional<File> getLog4JConfigurationFile() {
    List<File> files = new Log4J2ConfigurationLocator(log).locateFiles(new File(project.getBuild().getDirectory()));
    return files.isEmpty() ? Optional.empty() : Optional.of(files.get(0));
  }

  private Set<String> locateMunitTestSuites() {
    return new TestSuiteFilesLocator().locateFiles(munitSrcFolder).stream()
        .filter(suiteFile -> testSuiteFileFilter.shouldFilter(suiteFile.getName()))
        .map(suiteFile -> munitSrcFolder.toURI().relativize(suiteFile.toURI()).toASCIIString())
        .collect(Collectors.toSet());
  }

  private String getConfigResources() {
    MulePropertiesLoader propertiesLoader = new MulePropertiesLoader(project.getBuild().getOutputDirectory(),
                                                                     project.getBuild().getTestOutputDirectory(),
                                                                     log);
    return propertiesLoader.getConfigResources();
  }

  private String getRuntimeVersion() throws MojoExecutionException {
    if (StringUtils.isNotBlank(runtimeVersion)) {
      return runtimeVersion;
    }

    try {
      runtimeVersion = getMuleApplicationModel().getMinMuleVersion();
      log.debug("Runtime version to " + runtimeVersion + " obtained from the mule-application.json file");
      return runtimeVersion;
    } catch (IOException e) {
      String message = "Fail to obtain Runtime version from mule-application.json";
      log.error(message, e);
      throw new MojoExecutionException(message, e);
    }
  }

  private MuleApplicationModel getMuleApplicationModel() throws IOException {
    return new MuleApplicationModelJsonSerializer().deserialize(FileUtils.readFileToString(applicationJsonFile));
  }
}
