/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.munit.xml;

import com.thoughtworks.xstream.core.util.QuickWriter;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;
import com.thoughtworks.xstream.io.xml.PrettyPrintWriter;
import com.thoughtworks.xstream.io.xml.XppDriver;
import org.apache.commons.lang3.text.translate.NumericEntityEscaper;

import java.io.Writer;
import java.util.ArrayList;
import java.util.List;

public class CdataAwareXppDriver extends XppDriver {

  /**
   * List of field names that will be wrapped as CDATA
   */
  private static final List<String> CDATA_FIELDS = new ArrayList<String>();;

  static {
    /**
     * Wrap "failure" field to encapsulate StackTrace
     */
    CDATA_FIELDS.add("failure");
  }

  public CdataAwareXppDriver() {
    super();
  }

  /**
   * Override createWriter to wrap fields listed in CDATA_FIELDS with CDATA block
   */
  @Override
  public HierarchicalStreamWriter createWriter(Writer out) {
    return new PrettyPrintWriter(out) {

      boolean cdata = false;

      public void startNode(String name) {
        super.startNode(name);
        cdata = CDATA_FIELDS.contains(name);
      }

      protected void writeText(QuickWriter writer, String text) {
        if (cdata) {
          writer.write("<![CDATA[");
          writer.write(NumericEntityEscaper.below(0x20).translate(text));
          writer.write("]]>");
        } else {
          super.writeText(writer, text);
        }
      }
    };
  }
}
