package org.mule.runner.model;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.ToIntFunction;

import org.apache.commons.lang3.StringUtils;

public class RunResult {

  private SuiteResult currentSuite = null;
  private String stackTrace = StringUtils.EMPTY;

  // TODO disable due to MU-1091
  // private ApplicationCoverageReport coverageReport;
  private Map<String, List<String>> suiteOutputs = Collections.emptyMap();

  private boolean runFinished = false;
  private boolean coverageEnabled = true;
  private boolean unexpectedError = false;

  private Map<String, SuiteResult> suites = new HashMap<>();

  public void testSuiteStartFailed(String suiteFullPath, String stackTrace) {
    SuiteResult suite = new SuiteResult();
    suite.setRuntimeStartFailed(true);
    suite.setCause(stackTrace);
    suite.setSuiteFinished(true);
    suite.setSuitePath(suiteFullPath);
    suites.put(suiteFullPath, suite);
    suite.setTime(0);
  }

  public void testSuiteStarted(String suiteFullPath) {
    SuiteResult suite = new SuiteResult();
    suite.setSuitePath(suiteFullPath);
    suites.put(suiteFullPath, suite);
    currentSuite = suite;
  }

  public void testStarted(String suiteFullPath, String testName) {
    suites.get(suiteFullPath).add(testName, new TestResult(testName));
  }

  public void testFailure(String suiteFullPath, String testName, String stackTrace, long elapsedTime) {
    TestResult test = suites.get(suiteFullPath).getTest(testName);
    test.setFailed(true);
    test.setCause(stackTrace);
    test.setTime(elapsedTime);
  }

  public void testError(String suiteFullPath, String testName, String stackTrace, long elapsedTime) {
    TestResult test = suites.get(suiteFullPath).getTest(testName);
    test.setError(true);
    test.setCause(stackTrace);
    test.setTime(elapsedTime);
  }

  public void testFinished(String suiteFullPath, String testName, long elapsedTime) {
    TestResult test = suites.get(suiteFullPath).getTest(testName);
    test.setFinished(true);
    test.setTime(elapsedTime);
  }

  public void testIgnored(String suiteFullPath, String testName, long elapsedTime) {
    TestResult test = suites.get(suiteFullPath).getTest(testName);
    test.setIgnored(true);
    test.setTime(elapsedTime);
  }

  public void suiteFinished(String suiteFullPath, long elapsedTime) {
    suites.get(suiteFullPath).setSuiteFinished(true);
    currentSuite = null;
    suites.get(suiteFullPath).setTime(elapsedTime);
  }

  public void suiteFailure(String suiteFullPath, String stackTrace) {
    suites.get(suiteFullPath).setSuiteFailed(true);
    suites.get(suiteFullPath).setCause(stackTrace);
  }

  public void suiteError(String suiteFullPath, String stackTrace) {
    suites.get(suiteFullPath).setSuiteError(true);
    suites.get(suiteFullPath).setCause(stackTrace);
  }

  public void unexpectedError(String stackTrace) {
    currentSuite = null;
    unexpectedError = true;
    runFinished = true;
    this.stackTrace = stackTrace;
  }

  public Collection<SuiteResult> getSuites() {
    return suites.values();
  }

  public boolean isRunFinished() {
    return runFinished;
  }

  public void setRunFinished(boolean runFinished) {
    this.runFinished = runFinished;
  }

  public SuiteResult getSuite(String suiteName) {
    return suites.get(suiteName);
  }

  public int getNumberOfTests() {
    return summarize(SuiteResult::getNumberOfTests);
  }

  public int getNumberOfIgnores() {
    return summarize(SuiteResult::getNumberOfIgnores);
  }

  public int getNumberOfErrors() {
    return summarize(SuiteResult::getNumberOfErrors);
  }

  public int getNumberOfFailures() {
    return summarize(SuiteResult::getNumberOfFailures);
  }

  private int summarize(ToIntFunction<SuiteResult> predicate) {
    return getSuites().stream().mapToInt(predicate).sum();
  }

  public SuiteResult getCurrentSuite() {
    return currentSuite;
  }

  // TODO disable due to MU-1091
  // public ApplicationCoverageReport getApplicationCoverageReport() {
  // return coverageReport;
  // }
  //
  // public void setCoverageReport(String jsonReport) {
  // this.coverageReport = new Gson().fromJson(jsonReport, ApplicationCoverageReport.class);
  // }

  public Map<String, List<String>> getSuiteOutputs() {
    return suiteOutputs;
  }

  public void setSuiteOutputs(Map<String, List<String>> suiteOutputs) {
    this.suiteOutputs = suiteOutputs;
  }

  public boolean hasFailed() {
    return unexpectedError || suites.values().stream().anyMatch(suite -> (suite.hasFailed() || suite.hasError()));
  }

  public void setCoverageEnabled(boolean runCoverage) {
    this.coverageEnabled = runCoverage;
  }

  public boolean isCoverageEnabled() {
    return coverageEnabled;
  }

  public String getStackTrace() {
    return stackTrace;
  }
}
