package org.mule.runner.model;


import java.util.*;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

public class SuiteResult {

  private String suitePath;

  private boolean suiteFinished = false;
  private boolean runtimeStartFailed = false;

  private String cause;
  private Map<String, TestResult> tests = new HashMap<String, TestResult>();
  private long time;
  private boolean suiteFailed;
  private boolean suiteError;

  public void setRuntimeStartFailed(boolean runtimeStartFailed) {
    this.runtimeStartFailed = runtimeStartFailed;
  }

  public void setCause(String newCause) {
    if (StringUtils.isEmpty(cause)) {
      this.cause = newCause;
    }
  }

  public void add(String testName, TestResult testResult) {
    tests.put(testName, testResult);
  }

  public void setSuitePath(String suitePath) {
    this.suitePath = suitePath;
  }

  public void setSuiteFinished(boolean suiteFinished) {
    this.suiteFinished = suiteFinished;
  }

  public boolean isSuiteFinished() {
    return suiteFinished;
  }

  public boolean isRuntimeStartFailed() {
    return runtimeStartFailed;
  }

  public String getCause() {
    return cause;
  }

  public int getNumberOfProcessedTests() {
    return countTests(TestResult::isFinished);
  }

  public int getNumberOfIgnores() {
    return countTests(TestResult::isIgnored);
  }

  public int getNumberOfFailures() {
    return countTests(TestResult::hasFailed);
  }

  public int getNumberOfErrors() {
    return countTests(TestResult::hasError);
  }

  public int getNumberOfTests() {
    return tests.size();
  }

  public TestResult getTest(String string) {
    return tests.get(string);

  }

  public Optional<TestResult> getRunningTest() {
    return tests.values().stream().filter(test -> !test.isFinished()).findFirst();
  }

  public Collection<TestResult> getTests() {
    return tests.values();
  }

  public String getSuitePath() {
    return suitePath;
  }

  public boolean hasFailed() {
    return suiteFailed || findAnyTest(TestResult::hasFailed);
  }

  public boolean hasError() {
    return runtimeStartFailed || suiteError || findAnyTest(TestResult::hasError);
  }

  public boolean isIgnored() {
    return !tests.isEmpty() && tests.values().stream().allMatch(TestResult::isIgnored);
  }

  public List<TestResult> getFailingTests() {
    return filterTests(TestResult::hasFailed);
  }

  public List<TestResult> getErrorTests() {
    return filterTests(TestResult::hasError);
  }

  public void setTime(long time) {
    this.time = time;
  }

  public long getTime() {
    return time;
  }

  public void setSuiteFailed(boolean suiteFailed) {
    this.suiteFailed = suiteFailed;
  }

  public boolean isSuiteFailed() {
    return suiteFailed;
  }

  public void setSuiteError(boolean suiteError) {
    this.suiteError = suiteError;
  }

  public boolean isSuiteError() {
    return suiteError;
  }

  private int countTests(Predicate<TestResult> predicate) {
    return (int) tests.values().stream().filter(predicate).count();
  }

  private boolean findAnyTest(Predicate<TestResult> predicate) {
    return tests.values().stream().anyMatch(predicate);
  }

  private List<TestResult> filterTests(Predicate<TestResult> predicate) {
    return tests.values().stream().filter(predicate).collect(Collectors.toList());
  }
}
