package org.mule.runner.printer;


import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.logging.Log;
import org.mule.runner.model.RunResult;
import org.mule.runner.model.SuiteResult;
import org.mule.runner.model.TestResult;

public class MavenResultPrinter implements ResultPrinter {

  private static final String ERROR_TAG = "ERROR";
  private static final String FAILURE_TAG = "FAILED";
  private static final String SEPARATOR_TAG = " <<< ";
  private static final int MAX_STACKTRACE = 50;

  private Log log;

  public MavenResultPrinter(Log log) {
    this.log = log;
  }

  public void print(RunResult runResult) {

    log.info("====================================================================================");
    log.info("MUnit Run Summary                                                                   ");
    log.info("====================================================================================");

    int testCount = runResult.getNumberOfTests();
    int errorCount = runResult.getNumberOfErrors();
    int failCount = runResult.getNumberOfFailures();
    int skipCount = runResult.getNumberOfIgnores();

    for (SuiteResult suiteResult : runResult.getSuites()) {
      List<TestResult> failingTests = suiteResult.getFailingTests();
      List<TestResult> errorTests = suiteResult.getErrorTests();

      printSuiteResult(suiteResult);
      printFailures(failingTests, log);
      printError(errorTests, log);
    }

    log.info("\t");
    log.info("====================================================================================");
    log.info(" > Tests:   \t" + testCount);
    log.info(" > Errors:  \t" + errorCount);
    log.info(" > Failures:\t" + failCount);
    log.info(" > Skipped: \t" + skipCount);
    log.info("====================================================================================");
  }

  private void printSuiteResult(SuiteResult suiteResult) {
    String unsuccessfulTag = StringUtils.EMPTY;
    if (suiteResult.isSuiteFailed() || suiteResult.isSuiteError() || suiteResult.isRuntimeStartFailed()) {
      unsuccessfulTag = SEPARATOR_TAG;
      unsuccessfulTag += suiteResult.isSuiteFailed() ? FAILURE_TAG : ERROR_TAG;
      unsuccessfulTag += " (" + suiteResult.getCause().substring(0, MAX_STACKTRACE) + "...)";
    }
    String suiteResultString =
        (suiteResult.getNumberOfTests() == 0 ? " (No tests run)" : getSuiteResultSummary(suiteResult)) + unsuccessfulTag;
    log.info(" >> " + suiteResult.getSuitePath() + suiteResultString);
  }

  private void printFailures(List<TestResult> failingTests, Log log) {
    printUnsuccessfulTests(failingTests, FAILURE_TAG, log);
  }

  private void printError(List<TestResult> errorTests, Log log) {
    printUnsuccessfulTests(errorTests, ERROR_TAG, log);
  }

  private void printUnsuccessfulTests(List<TestResult> unsuccessfulTests, String unsuccessfulTag, Log log) {
    if (!unsuccessfulTests.isEmpty()) {
      for (TestResult result : unsuccessfulTests) {
        log.info("\t --- " + result.getTestName() + SEPARATOR_TAG + unsuccessfulTag);
      }
    }
  }

  private String getSuiteResultSummary(SuiteResult suiteResult) {
    return " test result: " +
        "Tests: " + suiteResult.getNumberOfTests() +
        ", Errors: " + suiteResult.getNumberOfErrors() +
        ", Failures:" + suiteResult.getNumberOfFailures() +
        ", Skipped: " + suiteResult.getNumberOfIgnores();
  }
}
