/*
 * Copyright (c) 2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master Subscription
 * Agreement (or other master license agreement) separately entered into in writing between
 * you and MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package org.mule.runner.printer;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.maven.plugin.logging.Log;
import org.mule.munit.xml.CdataAwareXppDriver;
import org.mule.munit.xml.Property;
import org.mule.munit.xml.TestCase;
import org.mule.munit.xml.TestSuite;
import org.mule.runner.model.RunResult;
import org.mule.runner.model.SuiteResult;
import org.mule.runner.model.TestResult;

import com.thoughtworks.xstream.XStream;

public class XmlResultPrinter extends FileResultPrinter {

  private static final float MILLIS_FACTOR = 1000.0f;
  private static final String XML_REPORT_HEADER = "TEST-munit.";

  private TestSuite suite;
  private String name;
  private Map<String, String> systemProperties;

  public XmlResultPrinter(File reportBasePath, Map<String, String> systemProperties, Log log) {
    super(reportBasePath, log);
    this.systemProperties = systemProperties;
  }

  public void print(RunResult runResult) {
    try {
      for (SuiteResult suiteResult : runResult.getSuites()) {
        printSuiteResult(suiteResult);
      }
    } catch (FileNotFoundException e) {
      e.printStackTrace();
    }
  }

  public void printSuiteResult(SuiteResult result) throws FileNotFoundException {
    CdataAwareXppDriver xppDriver = new CdataAwareXppDriver();
    XStream xStream = new XStream(xppDriver);
    xStream.autodetectAnnotations(true);

    this.name = FilenameUtils.getName(result.getSuitePath()).replace(".xml", StringUtils.EMPTY);
    this.suite = new TestSuite(dumpProperties(System.getProperties(), systemProperties), this.name);
    for (TestResult testResult : result.getTests()) {
      printTestResult(testResult);
    }
    suite.setErrors(result.getNumberOfErrors());
    suite.setFailures(result.getNumberOfFailures());
    suite.setTests(result.getNumberOfTests());
    suite.setTime(result.getTime() / MILLIS_FACTOR);
    suite.setSkipped(result.getNumberOfIgnores());

    String suiteRelativePath = result.getSuitePath();
    String suiteReportName = suiteRelativePath.replace("/", ".");
    PrintStream out = getResultPrintStream(XML_REPORT_HEADER + suiteReportName);
    out.print(xStream.toXML(suite));

  }

  public void printTestResult(TestResult testResult) {
    TestCase testCase = new TestCase(testResult.getTime() / MILLIS_FACTOR, name, testResult.getTestName());
    testCase.setSkipped(testResult.isIgnored());
    if (testResult.hasFailed()) {
      testCase.setFailure(testResult.getCause());
    }
    if (testResult.hasError()) {
      testCase.setError(testResult.getCause());
    }
    suite.add(testCase);
  }

  private List<Property> dumpProperties(Properties properties, Map<String, String> sysProps) {
    ArrayList<Property> testProperties = new ArrayList<>();
    for (Map.Entry<Object, Object> entry : properties.entrySet()) {
      if (!sysProps.containsKey(entry.getKey())) {
        testProperties.add(new Property((String) entry.getKey(), (String) entry.getValue()));
      }
    }
    for (Map.Entry<String, String> sysEntry : sysProps.entrySet()) {
      testProperties.add(new Property(sysEntry.getKey(), sysEntry.getValue()));
    }
    return testProperties;
  }

}
