package org.mule.util;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.plugin.logging.Log;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

public class ClasspathManager {

  private List<String> additionalClasspathElements;
  private List<String> classpathElements;

  private Set<Artifact> artifacts;
  private Class runnerClass;
  private Log log;

  public ClasspathManager(List<String> classpathElements,
                          List<String> additionalClasspathElements,
                          Set artifacts,
                          Class runnerClass,
                          Log log) {
    this.classpathElements = classpathElements;
    this.additionalClasspathElements = additionalClasspathElements;
    this.artifacts = artifacts;
    this.runnerClass = runnerClass;
    this.log = log;
  }

  public List<String> getEffectiveClasspath() {
    List<String> effectiveClasspath = new ArrayList<>();
    effectiveClasspath.addAll(classpathElements);
    addRunnerClassDependency(effectiveClasspath);
    addAdditionalClasspathElements(effectiveClasspath);
    log.debug("Classpath: ");
    for (String classpathEntry : effectiveClasspath) {
      log.debug(classpathEntry);
    }
    return effectiveClasspath;
  }

  private void addRunnerClassDependency(List<String> effectiveClasspath) {
    if (runnerClass != null) {
      String runnerClassDependency = runnerClass.getProtectionDomain().getCodeSource().getLocation().getPath();
      effectiveClasspath.add(runnerClassDependency);
      log.debug(String.format("Runner dependency added: %s", runnerClassDependency));
    } else {
      log.info("The runner class provided is null");
    }
  }

  private void addAdditionalClasspathElements(List<String> effectiveClasspath) {
    if (additionalClasspathElements != null) {
      for (String classpathEntry : additionalClasspathElements) {
        if (validateClasspathEntry(classpathEntry)) {
          effectiveClasspath.add(classpathEntry);
        } else {
          log.warn(String.format("Unable to add entry %s to classpath because it does not exist", classpathEntry));
        }
      }
    }
  }

  protected boolean validateClasspathEntry(String classpathEntry) {
    return new File(classpathEntry).exists();
  }

}
